package transform

import (
	"testing"
)

const (
	urlTestString = "Theskyabovetheportwasthecoloroftelevision,tunedtoadeadchannel.\xff\xff\xff\x3e\xfe\x00"
)

func TestURLEncodeString(t *testing.T) {
	encoded := URLEncodeString("foo !~")

	if encoded != "%66%6f%6f%20%21%7e" {
		t.Fatal(encoded)
	}

	t.Log(encoded)
}

func TestEncodeBase64(t *testing.T) {
	encoded := EncodeBase64("foo")

	if encoded != "Zm9v" {
		t.Fatal(encoded)
	}

	t.Log(encoded)
}

func TestEncodeBase64Chunks(t *testing.T) {
	chunks := EncodeBase64Chunks("1234567890123456789012345678901234567890", 10)
	expected := []string{"MTIzNDU2", "Nzg5MDEy", "MzQ1Njc4", "OTAxMjM0", "NTY3ODkw", "MTIzNDU2", "Nzg5MA=="}
	for i, c := range chunks {
		if c != expected[i] {
			t.Fatal(chunks)
		}
	}

	t.Log(chunks)

	chunks = EncodeBase64Chunks("1234567890123456789012345678901234567890", 12)
	expected = []string{"MTIzNDU2Nzg5", "MDEyMzQ1Njc4", "OTAxMjM0NTY3", "ODkwMTIzNDU2", "Nzg5MA=="}
	for i, c := range chunks {
		if c != expected[i] {
			t.Fatal(chunks)
		}
	}

	t.Log(chunks)

	chunks = EncodeBase64Chunks("1234567890123456789012345678901234567890", 13)
	for i, c := range chunks {
		if c != expected[i] {
			t.Fatal(chunks)
		}
	}

	t.Log(chunks)
}

func TestEncodeBase64Chunks_EmptyString(t *testing.T) {
	chunks := EncodeBase64Chunks("", 10)
	if len(chunks) != 0 {
		t.Fatal(len(""))
	}
}

func TestEncodeBase64Chunks_SmallerThanMaxsize(t *testing.T) {
	chunks := EncodeBase64Chunks("a", 10)
	if chunks[0] != "YQ==" || len(chunks) != 1 {
		t.Fatal(chunks)
	}
	t.Log(chunks)
}

func TestEncodeBase64URL(t *testing.T) {
	encoded := EncodeBase64URL(urlTestString)

	if encoded != "VGhlc2t5YWJvdmV0aGVwb3J0d2FzdGhlY29sb3JvZnRlbGV2aXNpb24sdHVuZWR0b2FkZWFkY2hhbm5lbC7___8-_gA=" {
		t.Fatal(encoded)
	}

	t.Log(encoded)
}

func TestDecodeBase64URL(t *testing.T) {
	decoded := DecodeBase64URL("VGhlc2t5YWJvdmV0aGVwb3J0d2FzdGhlY29sb3JvZnRlbGV2aXNpb24sdHVuZWR0b2FkZWFkY2hhbm5lbC7___8-_gA=")

	if decoded != urlTestString {
		t.Fatal(decoded)
	}

	t.Log(decoded)
}

func TestDecodeBase64URL_NoPad(t *testing.T) {
	decoded := DecodeBase64URL("VGhlc2t5YWJvdmV0aGVwb3J0d2FzdGhlY29sb3JvZnRlbGV2aXNpb24sdHVuZWR0b2FkZWFkY2hhbm5lbC7___8-_gA")

	if decoded != urlTestString {
		t.Fatal(decoded)
	}

	t.Log(decoded)
}

func TestDecodeBase64URL_WayTooMuchPadding(t *testing.T) {
	decoded := DecodeBase64URL("VGhlc2t5YWJvdmV0aGVwb3J0d2FzdGhlY29sb3JvZnRlbGV2aXNpb24sdHVuZWR0b2FkZWFkY2hhbm5lbC7___8-_gA=======")

	if decoded != urlTestString {
		t.Fatal(decoded)
	}

	t.Log(decoded)
}

func TestDecodeBase64(t *testing.T) {
	decoded := DecodeBase64("Zm9v")

	if decoded != "foo" {
		t.Fatal(decoded)
	}

	t.Log(decoded)
}

func TestTitle(t *testing.T) {
	title := Title("foo")

	if title != "Foo" {
		t.Fatal(title)
	}

	t.Log(title)
}

func TestPackLittleInt32(t *testing.T) {
	packed := PackLittleInt32(0x44434241)

	if packed != "ABCD" {
		t.Fatal(packed)
	}

	t.Log(packed)
}

func TestPackLittleInt64(t *testing.T) {
	packed := PackLittleInt64(0x4847464544434241)

	if packed != "ABCDEFGH" {
		t.Fatal(packed)
	}

	t.Log(packed)
}

func TestPackBigInt16(t *testing.T) {
	packed := PackBigInt16(0x4142)

	if packed != "AB" {
		t.Fatal(packed)
	}

	t.Log(packed)
}

func TestPackBigInt32(t *testing.T) {
	packed := PackBigInt32(0x41424344)

	if packed != "ABCD" {
		t.Fatal(packed)
	}

	t.Log(packed)
}

func TestPackBigInt64(t *testing.T) {
	packed := PackBigInt64(0x4142434445464748)

	if packed != "ABCDEFGH" {
		t.Fatal(packed)
	}

	t.Log(packed)
}

func TestInflate(t *testing.T) {
	compressed := "\x1f\x8b\x08\x08\xf4\xe9\x97\x66\x00\x03\x77\x61\x74\x00\x2b\x2b\xcd\xc9\x4b\xce\x48\x4d\xce\xe6\x02\x00\x3d\xf1\xb3\xf9\x0a\x00\x00\x00"
	inflated, ok := Inflate([]byte(compressed))
	if !ok {
		t.Fatal("Error decompressing data")
	}
	inflatedStr := string(inflated)
	if inflatedStr != "vulncheck\n" {
		t.Fatalf("Decompression generated unexpected string: %s", inflatedStr)
	}
}
