// Copyright 2023 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package civicinfo provides access to the Google Civic Information API.
//
// For product documentation, see: https://developers.google.com/civic-information/
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/civicinfo/v2"
//	...
//	ctx := context.Background()
//	civicinfoService, err := civicinfo.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for authentication.
//
// For information on how to create and obtain Application Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API keys), use option.WithAPIKey:
//
//	civicinfoService, err := civicinfo.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth flow), use option.WithTokenSource:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	civicinfoService, err := civicinfo.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See https://godoc.org/google.golang.org/api/option/ for details on options.
package civicinfo // import "google.golang.org/api/civicinfo/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint

const apiId = "civicinfo:v2"
const apiName = "civicinfo"
const apiVersion = "v2"
const basePath = "https://civicinfo.googleapis.com/"
const mtlsBasePath = "https://civicinfo.mtls.googleapis.com/"

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Divisions = NewDivisionsService(s)
	s.Elections = NewElectionsService(s)
	s.Representatives = NewRepresentativesService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Divisions *DivisionsService

	Elections *ElectionsService

	Representatives *RepresentativesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewDivisionsService(s *Service) *DivisionsService {
	rs := &DivisionsService{s: s}
	return rs
}

type DivisionsService struct {
	s *Service
}

func NewElectionsService(s *Service) *ElectionsService {
	rs := &ElectionsService{s: s}
	return rs
}

type ElectionsService struct {
	s *Service
}

func NewRepresentativesService(s *Service) *RepresentativesService {
	rs := &RepresentativesService{s: s}
	return rs
}

type RepresentativesService struct {
	s *Service
}

// AdministrationRegion: Describes information about a regional election
// administrative area.
type AdministrationRegion struct {
	// ElectionAdministrationBody: The election administration body for this
	// area.
	ElectionAdministrationBody *AdministrativeBody `json:"electionAdministrationBody,omitempty"`

	// LocalJurisdiction: The city or county that provides election
	// information for this voter. This object can have the same elements as
	// state.
	LocalJurisdiction *AdministrationRegion `json:"local_jurisdiction,omitempty"`

	// Name: The name of the jurisdiction.
	Name string `json:"name,omitempty"`

	// Sources: A list of sources for this area. If multiple sources are
	// listed the data has been aggregated from those sources.
	Sources []*Source `json:"sources,omitempty"`

	// ForceSendFields is a list of field names (e.g.
	// "ElectionAdministrationBody") to unconditionally include in API
	// requests. By default, fields with empty or default values are omitted
	// from API requests. However, any non-pointer, non-interface field
	// appearing in ForceSendFields will be sent to the server regardless of
	// whether the field is empty or not. This may be used to include empty
	// fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g.
	// "ElectionAdministrationBody") to include in API requests with the
	// JSON null value. By default, fields with empty values are omitted
	// from API requests. However, any field with an empty value appearing
	// in NullFields will be sent to the server as null. It is an error if a
	// field in this list has a non-empty value. This may be used to include
	// null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *AdministrationRegion) MarshalJSON() ([]byte, error) {
	type NoMethod AdministrationRegion
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// AdministrativeBody: Information about an election administrative body
// (e.g. County Board of Elections).
type AdministrativeBody struct {
	// AbsenteeVotingInfoUrl: A URL provided by this administrative body for
	// information on absentee voting.
	AbsenteeVotingInfoUrl string `json:"absenteeVotingInfoUrl,omitempty"`

	// BallotInfoUrl: A URL provided by this administrative body to give
	// contest information to the voter.
	BallotInfoUrl string `json:"ballotInfoUrl,omitempty"`

	// CorrespondenceAddress: The mailing address of this administrative
	// body.
	CorrespondenceAddress *SimpleAddressType `json:"correspondenceAddress,omitempty"`

	// ElectionInfoUrl: A URL provided by this administrative body for
	// looking up general election information.
	ElectionInfoUrl string `json:"electionInfoUrl,omitempty"`

	// ElectionNoticeText: A last minute or emergency notification text
	// provided by this administrative body.
	ElectionNoticeText string `json:"electionNoticeText,omitempty"`

	// ElectionNoticeUrl: A URL provided by this administrative body for
	// additional information related to the last minute or emergency
	// notification.
	ElectionNoticeUrl string `json:"electionNoticeUrl,omitempty"`

	// ElectionOfficials: The election officials for this election
	// administrative body.
	ElectionOfficials []*ElectionOfficial `json:"electionOfficials,omitempty"`

	// ElectionRegistrationConfirmationUrl: A URL provided by this
	// administrative body for confirming that the voter is registered to
	// vote.
	ElectionRegistrationConfirmationUrl string `json:"electionRegistrationConfirmationUrl,omitempty"`

	// ElectionRegistrationUrl: A URL provided by this administrative body
	// for looking up how to register to vote.
	ElectionRegistrationUrl string `json:"electionRegistrationUrl,omitempty"`

	// ElectionRulesUrl: A URL provided by this administrative body
	// describing election rules to the voter.
	ElectionRulesUrl string `json:"electionRulesUrl,omitempty"`

	// HoursOfOperation: A description of the hours of operation for this
	// administrative body.
	HoursOfOperation string `json:"hoursOfOperation,omitempty"`

	// Name: The name of this election administrative body.
	Name string `json:"name,omitempty"`

	// PhysicalAddress: The physical address of this administrative body.
	PhysicalAddress *SimpleAddressType `json:"physicalAddress,omitempty"`

	// VoterServices: A description of the services this administrative body
	// may provide.
	VoterServices []string `json:"voter_services,omitempty"`

	// VotingLocationFinderUrl: A URL provided by this administrative body
	// for looking up where to vote.
	VotingLocationFinderUrl string `json:"votingLocationFinderUrl,omitempty"`

	// ForceSendFields is a list of field names (e.g.
	// "AbsenteeVotingInfoUrl") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. However, any non-pointer, non-interface field appearing in
	// ForceSendFields will be sent to the server regardless of whether the
	// field is empty or not. This may be used to include empty fields in
	// Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AbsenteeVotingInfoUrl") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *AdministrativeBody) MarshalJSON() ([]byte, error) {
	type NoMethod AdministrativeBody
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Candidate: Information about a candidate running for elected office.
type Candidate struct {
	// CandidateUrl: The URL for the candidate's campaign web site.
	CandidateUrl string `json:"candidateUrl,omitempty"`

	// Channels: A list of known (social) media channels for this candidate.
	Channels []*Channel `json:"channels,omitempty"`

	// Email: The email address for the candidate's campaign.
	Email string `json:"email,omitempty"`

	// Name: The candidate's name. If this is a joint ticket it will
	// indicate the name of the candidate at the top of a ticket followed by
	// a / and that name of candidate at the bottom of the ticket. e.g.
	// "Mitt Romney / Paul Ryan"
	Name string `json:"name,omitempty"`

	// OrderOnBallot: The order the candidate appears on the ballot for this
	// contest.
	OrderOnBallot int64 `json:"orderOnBallot,omitempty,string"`

	// Party: The full name of the party the candidate is a member of.
	Party string `json:"party,omitempty"`

	// Phone: The voice phone number for the candidate's campaign office.
	Phone string `json:"phone,omitempty"`

	// PhotoUrl: A URL for a photo of the candidate.
	PhotoUrl string `json:"photoUrl,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CandidateUrl") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CandidateUrl") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Candidate) MarshalJSON() ([]byte, error) {
	type NoMethod Candidate
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Channel: A social media or web channel for a candidate.
type Channel struct {
	// Id: The unique public identifier for the candidate's channel.
	Id string `json:"id,omitempty"`

	// Type: The type of channel. The following is a list of types of
	// channels, but is not exhaustive. More channel types may be added at a
	// later time. One of: GooglePlus, YouTube, Facebook, Twitter
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Channel) MarshalJSON() ([]byte, error) {
	type NoMethod Channel
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Contest: Information about a contest that appears on a voter's
// ballot.
type Contest struct {
	// BallotPlacement: A number specifying the position of this contest on
	// the voter's ballot.
	BallotPlacement int64 `json:"ballotPlacement,omitempty,string"`

	// BallotTitle: The official title on the ballot for this contest, only
	// where available.
	BallotTitle string `json:"ballotTitle,omitempty"`

	// Candidates: The candidate choices for this contest.
	Candidates []*Candidate `json:"candidates,omitempty"`

	// District: Information about the electoral district that this contest
	// is in.
	District *ElectoralDistrict `json:"district,omitempty"`

	// ElectorateSpecifications: A description of any additional eligibility
	// requirements for voting in this contest.
	ElectorateSpecifications string `json:"electorateSpecifications,omitempty"`

	// Level: The levels of government of the office for this contest. There
	// may be more than one in cases where a jurisdiction effectively acts
	// at two different levels of government; for example, the mayor of the
	// District of Columbia acts at "locality" level, but also effectively
	// at both "administrative-area-2" and "administrative-area-1".
	//
	// Possible values:
	//   "international"
	//   "country"
	//   "administrativeArea1"
	//   "regional"
	//   "administrativeArea2"
	//   "locality"
	//   "subLocality1"
	//   "subLocality2"
	//   "special"
	Level []string `json:"level,omitempty"`

	// NumberElected: The number of candidates that will be elected to
	// office in this contest.
	NumberElected int64 `json:"numberElected,omitempty,string"`

	// NumberVotingFor: The number of candidates that a voter may vote for
	// in this contest.
	NumberVotingFor int64 `json:"numberVotingFor,omitempty,string"`

	// Office: The name of the office for this contest.
	Office string `json:"office,omitempty"`

	// PrimaryParties: If this is a partisan election, the name of the
	// party/parties it is for.
	PrimaryParties []string `json:"primaryParties,omitempty"`

	// PrimaryParty: [DEPRECATED] If this is a partisan election, the name
	// of the party it is for. This field as deprecated in favor of the
	// array "primaryParties", as contests may contain more than one party.
	PrimaryParty string `json:"primaryParty,omitempty"`

	// ReferendumBallotResponses: The set of ballot responses for the
	// referendum. A ballot response represents a line on the ballot. Common
	// examples might include "yes" or "no" for referenda. This field is
	// only populated for contests of type 'Referendum'.
	ReferendumBallotResponses []string `json:"referendumBallotResponses,omitempty"`

	// ReferendumBrief: Specifies a short summary of the referendum that is
	// typically on the ballot below the title but above the text. This
	// field is only populated for contests of type 'Referendum'.
	ReferendumBrief string `json:"referendumBrief,omitempty"`

	// ReferendumConStatement: A statement in opposition to the referendum.
	// It does not necessarily appear on the ballot. This field is only
	// populated for contests of type 'Referendum'.
	ReferendumConStatement string `json:"referendumConStatement,omitempty"`

	// ReferendumEffectOfAbstain: Specifies what effect abstaining (not
	// voting) on the proposition will have (i.e. whether abstaining is
	// considered a vote against it). This field is only populated for
	// contests of type 'Referendum'.
	ReferendumEffectOfAbstain string `json:"referendumEffectOfAbstain,omitempty"`

	// ReferendumPassageThreshold: The threshold of votes that the
	// referendum needs in order to pass, e.g. "two-thirds". This field is
	// only populated for contests of type 'Referendum'.
	ReferendumPassageThreshold string `json:"referendumPassageThreshold,omitempty"`

	// ReferendumProStatement: A statement in favor of the referendum. It
	// does not necessarily appear on the ballot. This field is only
	// populated for contests of type 'Referendum'.
	ReferendumProStatement string `json:"referendumProStatement,omitempty"`

	// ReferendumSubtitle: A brief description of the referendum. This field
	// is only populated for contests of type 'Referendum'.
	ReferendumSubtitle string `json:"referendumSubtitle,omitempty"`

	// ReferendumText: The full text of the referendum. This field is only
	// populated for contests of type 'Referendum'.
	ReferendumText string `json:"referendumText,omitempty"`

	// ReferendumTitle: The title of the referendum (e.g. 'Proposition 42').
	// This field is only populated for contests of type 'Referendum'.
	ReferendumTitle string `json:"referendumTitle,omitempty"`

	// ReferendumUrl: A link to the referendum. This field is only populated
	// for contests of type 'Referendum'.
	ReferendumUrl string `json:"referendumUrl,omitempty"`

	// Roles: The roles which this office fulfills.
	//
	// Possible values:
	//   "headOfState"
	//   "headOfGovernment"
	//   "deputyHeadOfGovernment"
	//   "governmentOfficer"
	//   "executiveCouncil"
	//   "legislatorUpperBody"
	//   "legislatorLowerBody"
	//   "highestCourtJudge"
	//   "judge"
	//   "schoolBoard"
	//   "specialPurposeOfficer"
	//   "otherRole"
	Roles []string `json:"roles,omitempty"`

	// Sources: A list of sources for this contest. If multiple sources are
	// listed, the data has been aggregated from those sources.
	Sources []*Source `json:"sources,omitempty"`

	// Special: "Yes" or "No" depending on whether this a contest being held
	// outside the normal election cycle.
	Special string `json:"special,omitempty"`

	// Type: The type of contest. Usually this will be 'General', 'Primary',
	// or 'Run-off' for contests with candidates. For referenda this will be
	// 'Referendum'. For Retention contests this will typically be
	// 'Retention'.
	Type string `json:"type,omitempty"`

	// ForceSendFields is a list of field names (e.g. "BallotPlacement") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "BallotPlacement") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Contest) MarshalJSON() ([]byte, error) {
	type NoMethod Contest
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DivisionSearchResponse: The result of a division search query.
type DivisionSearchResponse struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed
	// string "civicinfo#divisionSearchResponse".
	Kind string `json:"kind,omitempty"`

	Results []*DivisionSearchResult `json:"results,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Kind") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Kind") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DivisionSearchResponse) MarshalJSON() ([]byte, error) {
	type NoMethod DivisionSearchResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// DivisionSearchResult: Represents a political geographic division that
// matches the requested query.
type DivisionSearchResult struct {
	// Aliases: Other Open Civic Data identifiers that refer to the same
	// division -- for example, those that refer to other political
	// divisions whose boundaries are defined to be coterminous with this
	// one. For example, ocd-division/country:us/state:wy will include an
	// alias of ocd-division/country:us/state:wy/cd:1, since Wyoming has
	// only one Congressional district.
	Aliases []string `json:"aliases,omitempty"`

	// Name: The name of the division.
	Name string `json:"name,omitempty"`

	// OcdId: The unique Open Civic Data identifier for this division
	OcdId string `json:"ocdId,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Aliases") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Aliases") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *DivisionSearchResult) MarshalJSON() ([]byte, error) {
	type NoMethod DivisionSearchResult
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Election: Information about the election that was queried.
type Election struct {
	// ElectionDay: Day of the election in YYYY-MM-DD format.
	ElectionDay string `json:"electionDay,omitempty"`

	// Id: The unique ID of this election.
	Id int64 `json:"id,omitempty,string"`

	// Name: A displayable name for the election.
	Name string `json:"name,omitempty"`

	// OcdDivisionId: The political division of the election. Represented as
	// an OCD Division ID. Voters within these political jurisdictions are
	// covered by this election. This is typically a state such as
	// ocd-division/country:us/state:ca or for the midterms or general
	// election the entire US (i.e. ocd-division/country:us).
	OcdDivisionId string `json:"ocdDivisionId,omitempty"`

	// Possible values:
	//   "shapeLookupDefault"
	//   "shapeLookupDisabled"
	//   "shapeLookupEnabled"
	ShapeLookupBehavior string `json:"shapeLookupBehavior,omitempty"`

	// ForceSendFields is a list of field names (e.g. "ElectionDay") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "ElectionDay") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Election) MarshalJSON() ([]byte, error) {
	type NoMethod Election
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ElectionOfficial: Information about individual election officials.
type ElectionOfficial struct {
	// EmailAddress: The email address of the election official.
	EmailAddress string `json:"emailAddress,omitempty"`

	// FaxNumber: The fax number of the election official.
	FaxNumber string `json:"faxNumber,omitempty"`

	// Name: The full name of the election official.
	Name string `json:"name,omitempty"`

	// OfficePhoneNumber: The office phone number of the election official.
	OfficePhoneNumber string `json:"officePhoneNumber,omitempty"`

	// Title: The title of the election official.
	Title string `json:"title,omitempty"`

	// ForceSendFields is a list of field names (e.g. "EmailAddress") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "EmailAddress") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ElectionOfficial) MarshalJSON() ([]byte, error) {
	type NoMethod ElectionOfficial
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ElectionsQueryResponse: The list of elections available for this
// version of the API.
type ElectionsQueryResponse struct {
	// Elections: A list of available elections
	Elections []*Election `json:"elections,omitempty"`

	// Kind: Identifies what kind of resource this is. Value: the fixed
	// string "civicinfo#electionsQueryResponse".
	Kind string `json:"kind,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Elections") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Elections") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ElectionsQueryResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ElectionsQueryResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// ElectoralDistrict: Describes the geographic scope of a contest.
type ElectoralDistrict struct {
	// Id: An identifier for this district, relative to its scope. For
	// example, the 34th State Senate district would have id "34" and a
	// scope of stateUpper.
	Id string `json:"id,omitempty"`

	// Name: The name of the district.
	Name string `json:"name,omitempty"`

	// Scope: The geographic scope of this district. If unspecified the
	// district's geography is not known. One of: national, statewide,
	// congressional, stateUpper, stateLower, countywide, judicial,
	// schoolBoard, cityWide, township, countyCouncil, cityCouncil, ward,
	// special
	//
	// Possible values:
	//   "statewide"
	//   "congressional"
	//   "stateUpper"
	//   "stateLower"
	//   "countywide"
	//   "judicial"
	//   "schoolBoard"
	//   "citywide"
	//   "special"
	//   "countyCouncil"
	//   "township"
	//   "ward"
	//   "cityCouncil"
	//   "national"
	Scope string `json:"scope,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Id") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Id") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *ElectoralDistrict) MarshalJSON() ([]byte, error) {
	type NoMethod ElectoralDistrict
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// FeatureIdProto: A globally unique identifier associated with each
// feature. We use 128-bit identifiers so that we have lots of bits
// available to distinguish between features. The feature id currently
// consists of a 64-bit "cell id" that **sometimes** corresponds to the
// approximate centroid of the feature, plus a 64-bit fingerprint of
// other identifying information. See more on each respective field in
// its comments. Feature ids are first assigned when the data is created
// in MapFacts. After initial creation of the feature, they are
// immutable. This means that the only properties that you should rely
// on are that they are unique, and that cell_ids often - but not always
// - preserve spatial locality. The degree of locality varies as the
// feature undergoes geometry changes, and should not in general be
// considered a firm guarantee of the location of any particular
// feature. In fact, some locationless features have randomized cell
// IDs! Consumers of FeatureProtos from Mapfacts are guaranteed that
// fprints in the id field of features will be globally unique. Using
// the fprint allows consumers who don't need the spatial benefit of
// cell ids to uniquely identify features in a 64-bit address space.
// This property is not guaranteed for other sources of FeatureProtos.
type FeatureIdProto struct {
	// CellId: The S2CellId corresponding to the approximate location of
	// this feature as of when it was first created. This can be of variable
	// accuracy, ranging from the exact centroid of the feature at creation,
	// a very large S2 Cell, or even being completely randomized for
	// locationless features. Cell ids have the nice property that they
	// follow a space-filling curve over the surface of the earth. (See
	// s2cellid.h for details.) WARNING: Clients should only use cell IDs to
	// perform spatial locality optimizations. There is no strict guarantee
	// that the cell ID of a feature is related to the current geometry of
	// the feature in any way.
	CellId uint64 `json:"cellId,omitempty,string"`

	// Fprint: A 64-bit fingerprint used to identify features. Most clients
	// should rely on MapFacts or OneRing to choose fingerprints. If
	// creating new fprints, the strategy should be chosen so that the
	// chance of collision is remote or non-existent, and the distribution
	// should be reasonably uniform. For example, if the source data assigns
	// unique ids to features, then a fingerprint of the provider name,
	// version, and source id is sufficient.
	Fprint uint64 `json:"fprint,omitempty,string"`

	// TemporaryData: A place for clients to attach arbitrary data to a
	// feature ID. Never set in MapFacts.
	TemporaryData *MessageSet `json:"temporaryData,omitempty"`

	// ForceSendFields is a list of field names (e.g. "CellId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "CellId") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *FeatureIdProto) MarshalJSON() ([]byte, error) {
	type NoMethod FeatureIdProto
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// GeocodingSummary: Detailed summary of the result from geocoding an
// address
type GeocodingSummary struct {
	// AddressUnderstood: Represents the best estimate of whether or not the
	// input address was fully understood and the address is correctly
	// componentized. Mirrors the same-name field in
	// geostore.staging.AddressLinkupScoringProto.
	AddressUnderstood bool `json:"addressUnderstood,omitempty"`

	// FeatureId: The ID of the FeatureProto returned by the geocoder
	FeatureId *FeatureIdProto `json:"featureId,omitempty"`

	// FeatureType: The feature type for the FeatureProto returned by the
	// geocoder
	//
	// Possible values:
	//   "typeAny" - ABSTRACT
	//   "typeTransportation" - ABSTRACT
	//   "typeRoute" - A route is any section of road (or rails, etc.) that
	// has a name. This includes city streets as well as highways. Road
	// segments can belong to multiple routes (e.g. El Camino, CA-82).
	//   "typeDeprecatedHighwayDoNotUse" - DEPRECATED
	//   "typeHighway" - ABSTRACT
	//   "typeHighway1"
	//   "typeHighway2"
	//   "typeHighway3"
	//   "typeHighway4"
	//   "typeHighway5"
	//   "typeHighway6"
	//   "typeHighway7"
	//   "typeHighway8"
	//   "typeHighway9"
	//   "typeBicycleRoute" - A designated bicycle route, whose segments may
	// consist of any combination of bicycle paths, bicycle lanes, or city
	// streets.
	//   "typeTrail" - A designated trail, which may consist of paved
	// walkways, dirt paths, fire road, streets or highways, etc.
	//   "typeSegment" - ABSTRACT
	//   "typeRoad"
	//   "typeRailway" - Railroads use several different incompatible track
	// types.
	//   "typeStandardTrack"
	//   "typeJrTrack"
	//   "typeNarrowTrack"
	//   "typeMonorailTrack"
	//   "typeSubwayTrack"
	//   "typeLightRailTrack"
	//   "typeBroadTrack"
	//   "typeHighSpeedRail"
	//   "typeTrolleyTrack" - Tracks for streetcars, cable-cars, etc.
	// Ferries are services that are part of the road network but are not
	// roads. They typically involve fares and scheduled departure times.
	//   "typeFerry" - ABSTRACT
	//   "typeFerryBoat" - The vast majority of ferries are ferry boats.
	//   "typeFerryTrain" - Also called a "car transport", a ferry train is
	// a rail service that carries passengers and their vehicles across
	// undrivable terrain. The Channel Tunnel ("Chunnel") is the most famous
	// example, but they are also common in the Alps where they connect
	// neighboring valleys otherwise separated by impassable mountains.
	//   "typeVirtualSegment" - Any plausible 1-dimensional path through a
	// 2+ dimensional space, for the purposes of making graph-search-based
	// routing possible. Such segments can be used to model paths through
	// parking lots, squares, floors of buildings and other areas.
	//   "typeIntersection" - An intersection consists of a collection of
	// segments that terminate at the same location. This is topological
	// definition: it may not match what a typical user would think of as an
	// "intersection". See TYPE_INTERSECTION_GROUP, below, for more
	// information. Each segment terminating at an intersection has an
	// "endpoint type" that specifies how that segment is terminated: stop
	// sign, yield sign, three-way light, etc.
	//   "typeTransit" - ABSTRACT
	//   "typeTransitStation" - DEPRECATED
	//   "typeBusStation" - DEPRECATED
	//   "typeTramwayStation" - DEPRECATED
	//   "typeTrainStation" - DEPRECATED
	//   "typeSubwayStation" - DEPRECATED
	//   "typeFerryTerminal" - DEPRECATED
	//   "typeAirport" - DEPRECATED
	//   "typeAirportCivil" - DEPRECATED
	//   "typeAirportMilitary" - DEPRECATED
	//   "typeAirportMixed" - DEPRECATED
	//   "typeHeliport" - DEPRECATED
	//   "typeSeaplaneBase" - DEPRECATED
	//   "typeAirstrip" - DEPRECATED
	//   "typeCableCarStation" - DEPRECATED
	//   "typeGondolaLiftStation" - DEPRECATED
	//   "typeFunicularStation" - DEPRECATED
	//   "typeSpecialStation" - DEPRECATED
	//   "typeHorseCarriageStation" - DEPRECATED
	//   "typeMonorailStation" - DEPRECATED
	//   "typeSeaport" - DEPRECATED
	//   "typeTransitStop" - DEPRECATED
	//   "typeTransitTrip" - DEPRECATED
	//   "typeTransitDeparture" - DEPRECATED
	//   "typeTransitLeg" - DEPRECATED
	//   "typeTransitLine" - A transit line is a collection of transit legs,
	// associated with some invariant properties of the trips that run over
	// the legs. See also transitline.proto
	//   "typeTransitAgencyDeprecatedValue" - TYPE_TRANSIT_AGENCY was moved
	// to 0xC91. This deprecated enum value still exists for debugging
	// purposes only.
	//   "typeTransitTransfer" - DEPRECATED
	//   "typeSegmentPath" - ABSTRACT
	//   "typeRoadSign" - Road sign features have names, point geometry,
	// etc. They also have segment_path data (see below) which lists the
	// segments that refer to the sign. See segment.proto for the reference
	// from the segment to the road sign.
	//   "typeIntersectionGroup" - Our TYPE_INTERSECTION feature, above,
	// models the point where one or more segments terminate. This is
	// topological definition: it may not match what a typical user would
	// think of as an "intersection". Consider the intersections where
	// Hayes, Market, Larkin, and 9th Street meet near (37.77765,
	// -122.41638) in San Francisco. Most people would probably consider
	// this a single feature, even though we model it as four separate
	// TYPE_INTERSECTION features. This TYPE_INTERSECTION_GROUP is used to
	// model the user's concept of a complex intersection.
	//   "typePathway" - RESERVED
	//   "typeRestrictionGroup" - A restriction group describes a set of
	// segment restrictions that belong together and have a name or an
	// associated event. See also restriction_group.proto
	//   "typeTollCluster" - A toll cluster is either a single point on a
	// segment (represented as a point at the end of the segment that has
	// ENDPOINT_TOLL_BOOTH set) or a group of points on various road
	// segments in MapFacts that represents one or more lanes passing
	// through a toll fixture that all go to the same routing destination.
	// Each toll cluster should have at most a single price per payment
	// method. E.g. {CASH = $5, PASS = $1}. Note: If a toll fixture has
	// different prices for multiple routing destinations, drivers need to
	// be in the correct lane before passing through the toll fixture and
	// hence such a fixture is represented by multiple toll clusters. A toll
	// cluster does not necessarily represent a real-world entity, e.g. a
	// particular plaza/structure as perceived by humans. This is because a
	// plaza can be represented by more than one toll cluster. We require
	// toll clusters to have names, but they might be non-unique. For
	// example, a plaza might be represented by multiple toll clusters that
	// may have the same plaza name. For further details, please see
	// go/toll-cluster-schema.
	//   "typePolitical" - ABSTRACT
	//   "typeCountry"
	//   "typeAdministrativeArea" - ABSTRACT
	//   "typeAdministrativeArea1"
	//   "typeUsState" - DEPRECATED
	//   "typeGbCountry" - DEPRECATED
	//   "typeJpTodoufuken" - DEPRECATED
	//   "typeAdministrativeArea2"
	//   "typeGbFormerPostalCounty" - DEPRECATED
	//   "typeGbTraditionalCounty" - DEPRECATED
	//   "typeAdministrativeArea3"
	//   "typeAdministrativeArea4"
	//   "typeAdministrativeArea5"
	//   "typeAdministrativeArea6"
	//   "typeAdministrativeArea7"
	//   "typeAdministrativeArea8"
	//   "typeAdministrativeArea9"
	//   "typeColloquialArea" - e.g. Silicon Valley
	//   "typeReservation" - A reservation is a region collectively held or
	// governed by indigenous people and officially recognized by the
	// country’s government at the federal or state level. A reservation
	// may be fully contained within an administrative feature or partially
	// contained within two or more. These regions are referred to by
	// different categorical names depending on country and even by state,
	// including but not limited to: “Indian Reservations”, “Indian
	// Reserves”, “Land Claim Settlement Lands”, “Indian Lands”,
	// “Treaty Lands”, “Indigenous Territories”, etc. A reservation
	// is not a historic indigenous territory boundary or a region which has
	// applied for land rights but has not yet received official
	// recognition.
	//   "typeLocality"
	//   "typeGbPostTown" - DEPRECATED
	//   "typeJpGun" - DEPRECATED
	//   "typeJpShikuchouson" - DEPRECATED
	//   "typeJpSubShikuchouson" - DEPRECATED
	//   "typeColloquialCity" - An entity widely considered to be a city,
	// that may itself be made up of smaller political entities, some of
	// which are cities/towns/villages themselves. For example, the
	// colloquial view of Sydney, Australia actually comprises many smaller
	// cities, but is regarded as a city itself. This type is not suitable
	// for modeling official metro-/micropolitan or other statistical areas.
	//   "typeSublocality" - ABSTRACT
	//   "typeUsBorough" - DEPRECATED
	//   "typeGbDependentLocality" - DEPRECATED
	//   "typeJpOoaza" - DEPRECATED
	//   "typeJpKoaza" - DEPRECATED
	//   "typeJpGaiku" - DEPRECATED
	//   "typeGbDoubleDependentLocality" - DEPRECATED
	//   "typeJpChiban" - DEPRECATED
	//   "typeJpEdaban" - DEPRECATED
	//   "typeSublocality1"
	//   "typeSublocality2"
	//   "typeSublocality3"
	//   "typeSublocality4"
	//   "typeSublocality5"
	//   "typeNeighborhood"
	//   "typeConstituency"
	//   "typeDesignatedMarketArea" - Designated Market Areas (or DMAs) are
	// used by marketing and ratings companies (such as the Nielsen Media
	// Research company) to describe geographical regions (such as the
	// greater New York metropolitan area) that are covered by a set of
	// television stations. (See http://www.schooldata.com/pdfs/DMA.pdf) In
	// the United States, DMAs should have a DMA numeric ID name, tagged
	// with the FLAG_DESIGNATED_MARKET_AREA_ID flag.
	//   "typeSchoolDistrict"
	//   "typeLandParcel"
	//   "typeDisputedArea" - Eventually we'll have more data for disputed
	// areas (e.g., who makes claims on the area, who has de facto control,
	// etc.). For the moment, we just define a type so we can simply mark
	// areas as disputed.
	//   "typePoliceJurisdiction" - Boundaries representing the jurisdiction
	// of a particular police station.
	//   "typeStatisticalArea" - An area used for aggregating statistical
	// data, eg, a census region. Note that TYPE_STATISTICAL_AREA has a
	// third nibble so we can add an abstract parent above it later if need
	// be at 0x2E1 (and rename TYPE_STATISTICAL_AREA as
	// TYPE_STATISTICAL_AREA1).
	//   "typeConstituencyFuture" - DEPRECATED
	//   "typePark" - DEPRECATED
	//   "typeGolfCourse" - DEPRECATED
	//   "typeLocalPark" - DEPRECATED
	//   "typeNationalPark" - DEPRECATED
	//   "typeUsNationalPark" - DEPRECATED
	//   "typeUsNationalMonument" - DEPRECATED
	//   "typeNationalForest" - DEPRECATED
	//   "typeProvincialPark" - DEPRECATED
	//   "typeProvincialForest" - DEPRECATED
	//   "typeCampgrounds" - DEPRECATED
	//   "typeHikingArea" - DEPRECATED
	//   "typeBusiness" - DEPRECATED
	//   "typeGovernment" - DEPRECATED
	//   "typeBorderCrossing" - DEPRECATED
	//   "typeCityHall" - DEPRECATED
	//   "typeCourthouse" - DEPRECATED
	//   "typeEmbassy" - DEPRECATED
	//   "typeLibrary" - DEPRECATED
	//   "typeSchool" - DEPRECATED
	//   "typeUniversity" - DEPRECATED
	//   "typeEmergency" - DEPRECATED
	//   "typeHospital" - DEPRECATED
	//   "typePharmacy" - DEPRECATED
	//   "typePolice" - DEPRECATED
	//   "typeFire" - DEPRECATED
	//   "typeDoctor" - DEPRECATED
	//   "typeDentist" - DEPRECATED
	//   "typeVeterinarian" - DEPRECATED
	//   "typeTravelService" - DEPRECATED
	//   "typeLodging" - DEPRECATED
	//   "typeRestaurant" - DEPRECATED
	//   "typeGasStation" - DEPRECATED
	//   "typeParking" - DEPRECATED
	//   "typePostOffice" - DEPRECATED
	//   "typeRestArea" - DEPRECATED
	//   "typeCashMachine" - DEPRECATED
	//   "typeCarRental" - DEPRECATED
	//   "typeCarRepair" - DEPRECATED
	//   "typeShopping" - DEPRECATED
	//   "typeGrocery" - DEPRECATED
	//   "typeTouristDestination" - DEPRECATED
	//   "typeEcoTouristDestination" - DEPRECATED
	//   "typeBirdWatching" - DEPRECATED
	//   "typeFishing" - DEPRECATED
	//   "typeHunting" - DEPRECATED
	//   "typeNatureReserve" - DEPRECATED
	//   "typeTemple" - DEPRECATED
	//   "typeChurch" - DEPRECATED
	//   "typeGurudwara" - DEPRECATED
	//   "typeHinduTemple" - DEPRECATED
	//   "typeMosque" - DEPRECATED
	//   "typeSynagogue" - DEPRECATED
	//   "typeStadium" - DEPRECATED
	//   "typeBar" - DEPRECATED
	//   "typeMovieRental" - DEPRECATED
	//   "typeCoffee" - DEPRECATED
	//   "typeGolf" - DEPRECATED
	//   "typeBank" - DEPRECATED
	//   "typeDoodle" - DEPRECATED
	//   "typeGrounds" - DEPRECATED
	//   "typeAirportGrounds" - DEPRECATED
	//   "typeBuildingGrounds" - DEPRECATED
	//   "typeCemetery" - DEPRECATED
	//   "typeHospitalGrounds" - DEPRECATED
	//   "typeIndustrial" - DEPRECATED
	//   "typeMilitary" - DEPRECATED
	//   "typeShoppingCenter" - DEPRECATED
	//   "typeSportsComplex" - DEPRECATED
	//   "typeUniversityGrounds" - DEPRECATED
	//   "typeDeprecatedTarmac" - DEPRECATED
	//   "typeEnclosedTrafficArea" - DEPRECATED
	//   "typeParkingLot" - DEPRECATED
	//   "typeParkingGarage" - DEPRECATED
	//   "typeOffRoadArea" - DEPRECATED
	//   "typeBorder" - A line representing the boundary between two
	// features. See border.proto for details.
	//   "typeBuilding" - DEPRECATED
	//   "typeGeocodedAddress" - An association of a point with an address,
	// with no other information.
	//   "typeNaturalFeature" - ABSTRACT
	//   "typeTerrain" - Expanses of land that share common surface
	// attributes. These areas would look more or less uniform from a high
	// altitude.
	//   "typeSand"
	//   "typeBeach"
	//   "typeDune"
	//   "typeRocky"
	//   "typeIce"
	//   "typeGlacier"
	//   "typeBuiltUpArea" - Terrain that looks populated.
	//   "typeVegetation" - Terrain that is covered in vegetation.
	//   "typeShrubbery"
	//   "typeWoods"
	//   "typeAgricultural"
	//   "typeGrassland"
	//   "typeTundra"
	//   "typeDesert"
	//   "typeSaltFlat" - A flat expanse of salt left by the evaporation of
	// a body of salt water.
	//   "typeWater" - Features can be TYPE_WATER if we don't have enough
	// information to properly type the body of water. TYPE_WATER is also
	// used as the type for child features that compose a TYPE_RIVER
	// feature.
	//   "typeOcean" - One of the large salt-water bodies that covers most
	// of the globe.
	//   "typeBay" - An ocean subdivision formed by a coastal indentation.
	// Includes coves and gulfs.
	//   "typeBight" - An open body of water formed by a slight coastal
	// indentation.
	//   "typeLagoon"
	//   "typeSea" - An ocean subdivision more or less confined by land and
	// islands.
	//   "typeStrait" - A long narrow ocean subdivision. Includes sounds.
	//   "typeInlet"
	//   "typeFjord"
	//   "typeLake" - An inland body of standing water.
	//   "typeSeasonalLake" - A lake that dries up part of the year.
	//   "typeReservoir" - An artificial body of water, possibly created by
	// a dam, often used for irrigation or house use.
	//   "typePond"
	//   "typeRiver" - An inland body of moving water, or parts associated
	// with it in which there is little or no current (backwater).
	//   "typeRapids"
	//   "typeDistributary" - A branch which flows away from the main river.
	// Includes deltas.
	//   "typeConfluence" - A place where two or more rivers join.
	//   "typeWaterfall"
	//   "typeSpring" - A place where ground water flows naturally out of
	// the ground.
	//   "typeGeyser"
	//   "typeHotSpring"
	//   "typeSeasonalRiver" - A river that dries up part of the year.
	//   "typeWadi" - A dry riverbed that occasionally receives flashfloods.
	//   "typeEstuary" - A place at the end of a river where fresh and salt
	// water mix. Includes tidal creeks and limans.
	//   "typeWetland" - Land that is usually flooded. Includes bogs,
	// marshes, flats, moors, and swamps.
	//   "typeWaterNavigation"
	//   "typeFord" - A shallow place where water may be waded through.
	//   "typeCanal" - A narrow passage used by boats. Normally artificial.
	//   "typeHarbor" - A deep place near a shore where ships commonly drop
	// anchor.
	//   "typeChannel" - A deep part in a body of water that is suitable for
	// navigation. Includes narrows.
	//   "typeReef" - Rocks, coral, sandbars, or other features beneath the
	// surface of the water that pose a hazard to passing ships. Includes
	// shoals.
	//   "typeReefFlat" - A relatively shallow zone of the back reef located
	// closest to the shore, that may be exposed at low tide.
	//   "typeReefGrowth" - A small section of rocks, coral, sandbars, or
	// other features beneath the surface of the water that forms part of a
	// reef.
	//   "typeReefExtent" - The full extent of the reef complex.
	//   "typeReefRockSubmerged" - A submerged rock in the water.
	//   "typeIrrigation" - Man-made (and sometimes natural) channels used
	// to move water. This type was used for both dam structures and water
	// that is hold back by dams. We should use TYPE_COMPOUND_BUILDING for
	// dam structures and TYPE_RESERVOIR for water.
	//   "typeDam" - DEPRECATED
	//   "typeDrinkingWater"
	//   "typeCurrent" - Includes overfalls.
	//   "typeWateringHole" - A natural depression filled with water where
	// animals come to drink.
	//   "typeTectonic" - ABSTRACT This type is incorrectly under
	// TYPE_TECTONIC instead of TYPE_WATER. This was a mistake and is now
	// fixed. See TYPE_WATERING_HOLE for the replacement.
	//   "typeWateringHoleDeprecated" - DEPRECATED
	//   "typeVolcano"
	//   "typeLavaField"
	//   "typeFissure"
	//   "typeFault"
	//   "typeLandMass"
	//   "typeContinent"
	//   "typeIsland"
	//   "typeAtoll"
	//   "typeOceanRockExposed" - An exposed rock in the water.
	//   "typeCay" - A small, low-elevation, sandy island formed on the
	// surface of coral reefs
	//   "typePeninsula" - A stretch of land projecting into water. Includes
	// capes and spits.
	//   "typeIsthmus" - A strip of land connecting two larger land masses,
	// such as continents.
	//   "typeElevated" - Features that are notable for being high (or low),
	// or for having sudden changes in elevation. These features might have
	// an "elevation" extension to specify the actual elevation. See
	// ElevationProto for more information.
	//   "typePeak" - Elevations that have a distinctive peak.
	//   "typeNunatak" - A peak or ridge of a mountain that extends through
	// a glacier.
	//   "typeSpur" - A subsidiary peak of a mountain.
	//   "typePass" - A route over an otherwise difficult to traverse
	// feature. Includes saddle.
	//   "typePlateau" - Elevations that are flat on top. Includes mesas and
	// buttes.
	//   "typeRidge" - A ridge is a geographical feature consisting of a
	// chain of mountains or hills that form a continuous elevated crest
	// with a single ridgeline for some distance.
	//   "typeRavine" - Steep declines usually carved by erosion. Includes
	// valleys, canyons, ditches, and gorges.
	//   "typeCrater" - Depressions causes by impact, explosion, and
	// sometimes sink-holes.
	//   "typeKarst" - Topography formed on limestone and gypsum by
	// dissolution with sinkholes, caves, etc.
	//   "typeCliff" - A vertical or nearly vertical slope. Includes
	// escarpments.
	//   "typeVista" - An elevated place that is notable for having a good
	// view. Raster digital elevation data. This is not a type to be used by
	// providers or consumed by clients.
	//   "typeDigitalElevationModel" - RESERVED
	//   "typeUpland" - Land along streams higher than the alluvial plain or
	// stream terrace.
	//   "typeTerrace"
	//   "typeSlope" - Land not so steep as a cliff, but changing elevation.
	// Includes slides.
	//   "typeContourLine" - All the points on the polygon are at the same
	// elevation.
	//   "typePan" - A near-level shallow, natural depression or basin,
	// usually containing an intermittent lake, pond, or pool.
	//   "typeUnstableHillside"
	//   "typeMountainRange" - A series of mountains or hills ranged in a
	// line and connected by high ground. Mountain ranges usually consist of
	// many smaller ridges. For example, the Himalayas, the Andes. the Alps,
	// etc.
	//   "typeUndersea" - Features that are notable for being high (or low),
	// or for having sudden changes in elevation. These features might have
	// an "elevation" extension to specify the actual elevation. See
	// ElevationProto for more information.
	//   "typeSubmarineSeamount" - includes peaks, ranges, and spurs
	//   "typeSubmarineRidge"
	//   "typeSubmarineGap" - includes saddles
	//   "typeSubmarinePlateau"
	//   "typeSubmarineDeep"
	//   "typeSubmarineValley" - includes trenches and troughs
	//   "typeSubmarineBasin"
	//   "typeSubmarineSlope"
	//   "typeSubmarineCliff"
	//   "typeSubmarinePlain"
	//   "typeSubmarineFractureZone"
	//   "typeCave" - Don't use 0xA7. Use 8 bits for additional types under
	// TYPE_NATURAL_FEATURE, so we don't run out of space. The following are
	// miscellaneous natural features that don't fit any of the categories
	// above.
	//   "typeRock"
	//   "typeArchipelago" - A feature representing a group or chain of
	// islands.
	//   "typePostal" - ABSTRACT
	//   "typePostalCode" - This is the type for postal codes which are
	// complete and independent enough that there should be a feature for
	// them (e.g. US 5-digit ZIP codes). For even more detailed suffixes
	// that further subdivide a postal code (such as the +4 component in US
	// ZIP codes), store the information in a TYPE_POSTAL_CODE_SUFFIX
	// address component. When a range or set of postal codes share the same
	// geographical area, e.g. because a precise subdivision does not exist
	// or this subdivision is unknown, this type is used for each individual
	// postal code.
	//   "typePostalCodePrefix" - A prefix portion of a postal code which
	// does not meet the requirements for TYPE_POSTAL_CODE, but which is
	// useful to search for, for example UK outcodes.
	//   "typePremise" - DEPRECATED
	//   "typeSubPremise" - DEPRECATED This is deprecated and we want to use
	// TYPE_COMPOUND_SECTION instead.
	//   "typeSuite" - DEPRECATED
	//   "typePostTown" - The term "post town" is used for a
	// locality-like-entity that is only used for postal addresses.
	//   "typePostalRound" - DEPRECATED
	//   "typeMetaFeature" - ABSTRACT
	//   "typeDataSource" - Every data source used in constructing a data
	// repository has a corresponding feature that provides more information
	// about that data source. The extra information is stored in the
	// optional data_source field below.
	//   "typeLocale" - A locale feature provides region specific
	// conventions such as preferred language and formatting details for
	// time, date, and currency values. Locales aren't necessary defined by
	// physical geographic features, so they are classified as
	// meta-features.
	//   "typeTimezone" - A timezone feature is used to specify the region
	// covering an international timezone. When a point is covered by
	// multiple timezone features, the most specific one can be used to
	// compute the local time at this point. Most specific implies a much
	// smaller region or the one that is closer to the center. A feature's
	// timezone can be specified in the repeated related_timezone field.
	//   "typeBusinessChain" - A business chain feature is used to represent
	// a chain, e.g. Starbucks, McDonald's, etc. Other features representing
	// specific stores/franchises of this chain may refer to one such
	// feature via RELATION_MEMBER_OF_CHAIN. This is not strictly reserved
	// to commercial chains but can also be used to model organizations such
	// as the Red Cross or the United Nations.
	//   "typePhoneNumberPrefix" - A phone number prefix feature is used to
	// specify the region where phone numbers (typically fixed-line numbers)
	// must begin with a certain prefix. Any phone number prefix down to any
	// level of granularity could be represented by this type.
	//   "typePhoneNumberAreaCode" - A phone number area code is a prefix
	// which also coincides with the area code, or national destination
	// code, of a particular region.
	//   "typeBusinessCorridor" - A Business Corridor is a dense cluster of
	// semantically similar establishments. TYPE_BUSINESS_CORRIDOR features
	// are distinguished from TYPE_COLLOQUIAL_AREA features because the
	// corridors are not under the political hierarchy, are allowed to be
	// nameless, and may not correspond to well-known real world locations.
	// For more details, see go/geo-corridors-schema.
	//   "typeAddressTemplate" - An address template feature provides
	// region-specific conventions for structuring addresses. These features
	// aren't necessarily defined by physical geographic features, so they
	// are classified as meta-features.
	//   "typeTransitAgency" - A transit agency operates a number of lines,
	// typically all in the same city, region or country. See also
	// transitagency.proto
	//   "typeFutureGeometry" - A feature whose geometry is planned to
	// replace the geometry on another feature.
	//   "typeEvent" - DEPRECATED
	//   "typeEarthquake" - DEPRECATED
	//   "typeHurricane" - DEPRECATED
	//   "typeWeatherCondition" - DEPRECATED
	//   "typeTransient" - RESERVED
	//   "typeEntrance" - A portal of entry or exit to another feature.
	// Examples: - Subway station entrance. - Parking lot entrance.
	//   "typeCartographic" - Cartographic features are used to capture
	// real-world objects for which there is no current desire to model any
	// specific attributes. These are only useful to make the map tiles look
	// pretty.
	//   "typeHighTension" - DEPRECATED
	//   "typeSkiTrail" - Also see skitrail.proto
	//   "typeSkiLift" - Also see skilift.proto
	//   "typeSkiBoundary" - Also see skiboundary.proto
	//   "typeWatershedBoundary"
	//   "typeTarmac" - Starting with TYPE_TARMAC, we use longer IDs, so
	// that we can expand the number of feature types under
	// TYPE_CARTOGRAPHIC.
	//   "typeWall" - Use TYPE_COMPOUND_GROUND and appropriate gcids for the
	// next two.
	//   "typePicnicArea" - DEPRECATED
	//   "typePlayGround" - DEPRECATED
	//   "typeTrailHead"
	//   "typeGolfTeeingGround" - Sub-types within a golf course.
	//   "typeGolfPuttingGreen"
	//   "typeGolfRough"
	//   "typeGolfSandBunker"
	//   "typeGolfFairway"
	//   "typeGolfHole" - Use TYPE_ESTABLISHMENT_POI and gcid:golf_shop for
	// golf shops instead.
	//   "typeDeprecatedGolfShop" - DEPRECATED
	//   "typeCampingSite" - DEPRECATED
	//   "typeDesignatedBarbecuePit"
	//   "typeDesignatedCookingArea"
	//   "typeCampfirePit"
	//   "typeWaterFountain"
	//   "typeLitterReceptacle"
	//   "typeLockerArea"
	//   "typeAnimalEnclosure" - Subtype within a zoo - a cage or fenced-off
	// or otherwise delineated area containing animals.
	//   "typeCartographicLine" - A line for a cartographic detail. For
	// example the international date line. Such features should have
	// polyline geometry.
	//   "typeEstablishment" - ABSTRACT This type is being replaced by
	// TYPE_COMPOUND_GROUNDS. For further details, see go/compounds-v2
	//   "typeEstablishmentGrounds" - DEPRECATED This type has been replaced
	// by TYPE_COMPOUND_BUILDING. For further details, see
	// go/oyster-compounds
	//   "typeEstablishmentBuilding" - DEPRECATED
	//   "typeEstablishmentPoi" - An establishment which has a address
	// (a.k.a. location or storefront). Note that it *may* also have a
	// service area (e.g. a restaurant that offers both dine-in and
	// delivery). This type of business is also known as a "hybrid" Service
	// Area Business. Establishment POIs can be referenced by TYPE_COMPOUND
	// features using the RELATION_PRIMARILY_OCCUPIED_BY. This is the
	// reciprocal relation of the RELATION_OCCUPIES.
	//   "typeEstablishmentService" - A business without a storefront, e.g.
	// a plumber. It would normally not have a place that a customer could
	// visit to receive service, but it would have an area served by the
	// business. Also known as a "pure" Service Area Business. NOTE(tcain):
	// Using value 0xD441, since we could find ourselves with a need to
	// differentiate service areas from online-only at this level in the
	// future, but still benefit from being able to group those under a
	// common parent, disjoint from TYPE_ESTABLISHMENT_POI.
	//   "typeCelestial" - The root of types of features that are in the
	// sky, rather than on the earth. There will eventually be a hierarchy
	// of types here.
	//   "typeRoadMonitor" - Features responsible for monitoring traffic on
	// roads (usually for speed). Includes cameras at particular points as
	// well as monitors that cover larger spans. Features of this type
	// should have a corresponding gcid that specifies the correct subtype
	// (e.g. gcid:road_camera or gcid:speed_camera_zone). This type was
	// originally named as TYPE_ROAD_CAMERA.
	//   "typePublicSpacesAndMonuments" - ABSTRACT
	//   "typeStatue" - Note that this type does not distinguish the nature
	// of the statue (religious, historical, memorial, tourist, ...).
	//   "typeTownSquare" - Open space used for events, gathering, or as
	// market-place.
	//   "typeLevel" - A feature used to represent a logical level, e.g.
	// floor.
	//   "typeCompound" - ABSTRACT
	//   "typeCompoundGrounds" - e.g. campus, compound, parcel.
	//   "typeCompoundBuilding" - e.g. single family dwelling, office
	// building.
	//   "typeCompoundSection" - e.g. suite, room, hallway, cubicle.
	//   "typeTerminalPoint" - A terminal point represents a good location
	// for a user to meet a taxi, ridesharing vehicle, or general driver.
	//   "typeRegulatedArea" - An area controlled in some way by an
	// authoritative source, such as a government-designated COVID
	// containment zone or an area under government sanctions. Features of
	// this type should have one or more gcids corresponding to their
	// specific regulation, and client handling of these features may vary
	// based on the type of regulation.
	//   "typeLogicalBorder" - A grouping of TYPE_BORDER features ("border
	// segments"), which together represent a border between two features of
	// the same type.
	//   "typeDoNotUseReservedToCatchGeneratedFiles" - DEPRECATED
	//   "typeUnknown" - A feature of completely unknown type. This should
	// only be used when absolutely necessary. One example in which this
	// type is useful is in the Chinese importer, which must heuristically
	// segment addresses into components - it often does not know what types
	// to make those components. Please note that the Oyster address
	// formatter does not currently support address components of
	// TYPE_UNKNOWN well.
	FeatureType string `json:"featureType,omitempty"`

	// PositionPrecisionMeters: Precision of the center point (lat/long) of
	// the geocoded FeatureProto
	PositionPrecisionMeters float64 `json:"positionPrecisionMeters,omitempty"`

	// QueryString: The query sent to the geocoder
	QueryString string `json:"queryString,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AddressUnderstood")
	// to unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AddressUnderstood") to
	// include in API requests with the JSON null value. By default, fields
	// with empty values are omitted from API requests. However, any field
	// with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *GeocodingSummary) MarshalJSON() ([]byte, error) {
	type NoMethod GeocodingSummary
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *GeocodingSummary) UnmarshalJSON(data []byte) error {
	type NoMethod GeocodingSummary
	var s1 struct {
		PositionPrecisionMeters gensupport.JSONFloat64 `json:"positionPrecisionMeters"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.PositionPrecisionMeters = float64(s1.PositionPrecisionMeters)
	return nil
}

// GeographicDivision: Describes a political geography.
type GeographicDivision struct {
	// AlsoKnownAs: Any other valid OCD IDs that refer to the same
	// division.\n\nBecause OCD IDs are meant to be human-readable and at
	// least somewhat predictable, there are occasionally several
	// identifiers for a single division. These identifiers are defined to
	// be equivalent to one another, and one is always indicated as the
	// primary identifier. The primary identifier will be returned in ocd_id
	// above, and any other equivalent valid identifiers will be returned in
	// this list.\n\nFor example, if this division's OCD ID is
	// ocd-division/country:us/district:dc, this will contain
	// ocd-division/country:us/state:dc.
	AlsoKnownAs []string `json:"alsoKnownAs,omitempty"`

	// Name: The name of the division.
	Name string `json:"name,omitempty"`

	// OfficeIndices: List of indices in the offices array, one for each
	// office elected from this division. Will only be present if
	// includeOffices was true (or absent) in the request.
	OfficeIndices []int64 `json:"officeIndices,omitempty"`

	// ForceSendFields is a list of field names (e.g. "AlsoKnownAs") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AlsoKnownAs") to include
	// in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. However, any field with
	// an empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *GeographicDivision) MarshalJSON() ([]byte, error) {
	type NoMethod GeographicDivision
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// MessageSet: This is proto2's version of MessageSet.
type MessageSet struct {
}

// Office: Information about an Office held by one or more Officials.
type Office struct {
	// DivisionId: The OCD ID of the division with which this office is
	// associated.
	DivisionId string `json:"divisionId,omitempty"`

	// Levels: The levels of government of which this office is part. There
	// may be more than one in cases where a jurisdiction effectively acts
	// at two different levels of government; for example, the mayor of the
	// District of Columbia acts at "locality" level, but also effectively
	// at both "administrative-area-2" and "administrative-area-1".
	//
	// Possible values:
	//   "international"
	//   "country"
	//   "administrativeArea1"
	//   "regional"
	//   "administrativeArea2"
	//   "locality"
	//   "subLocality1"
	//   "subLocality2"
	//   "special"
	Levels []string `json:"levels,omitempty"`

	// Name: The human-readable name of the office.
	Name string `json:"name,omitempty"`

	// OfficialIndices: List of indices in the officials array of people who
	// presently hold this office.
	OfficialIndices []int64 `json:"officialIndices,omitempty"`

	// Roles: The roles which this office fulfills. Roles are not meant to
	// be exhaustive, or to exactly specify the entire set of
	// responsibilities of a given office, but are meant to be rough
	// categories that are useful for general selection from or sorting of a
	// list of offices.
	//
	// Possible values:
	//   "headOfState"
	//   "headOfGovernment"
	//   "deputyHeadOfGovernment"
	//   "governmentOfficer"
	//   "executiveCouncil"
	//   "legislatorUpperBody"
	//   "legislatorLowerBody"
	//   "highestCourtJudge"
	//   "judge"
	//   "schoolBoard"
	//   "specialPurposeOfficer"
	//   "otherRole"
	Roles []string `json:"roles,omitempty"`

	// Sources: A list of sources for this office. If multiple sources are
	// listed, the data has been aggregated from those sources.
	Sources []*Source `json:"sources,omitempty"`

	// ForceSendFields is a list of field names (e.g. "DivisionId") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "DivisionId") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Office) MarshalJSON() ([]byte, error) {
	type NoMethod Office
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Official: Information about a person holding an elected office.
type Official struct {
	// Address: Addresses at which to contact the official.
	Address []*SimpleAddressType `json:"address,omitempty"`

	// Channels: A list of known (social) media channels for this official.
	Channels []*Channel `json:"channels,omitempty"`

	// Emails: The direct email addresses for the official.
	Emails []string `json:"emails,omitempty"`

	// GeocodingSummaries: Detailed summary about the official's address's
	// geocoding
	GeocodingSummaries []*GeocodingSummary `json:"geocodingSummaries,omitempty"`

	// Name: The official's name.
	Name string `json:"name,omitempty"`

	// Party: The full name of the party the official belongs to.
	Party string `json:"party,omitempty"`

	// Phones: The official's public contact phone numbers.
	Phones []string `json:"phones,omitempty"`

	// PhotoUrl: A URL for a photo of the official.
	PhotoUrl string `json:"photoUrl,omitempty"`

	// Urls: The official's public website URLs.
	Urls []string `json:"urls,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Address") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Address") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Official) MarshalJSON() ([]byte, error) {
	type NoMethod Official
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// PollingLocation: A location where a voter can vote. This may be an
// early vote site, an election day voting location, or a drop off
// location for a completed ballot.
type PollingLocation struct {
	// Address: The address of the location.
	Address *SimpleAddressType `json:"address,omitempty"`

	// EndDate: The last date that this early vote site or drop off location
	// may be used. This field is not populated for polling locations.
	EndDate string `json:"endDate,omitempty"`

	// Latitude: Latitude of the location, in degrees north of the equator.
	// Note this field may not be available for some locations.
	Latitude float64 `json:"latitude,omitempty"`

	// Longitude: Longitude of the location, in degrees east of the Prime
	// Meridian. Note this field may not be available for some locations.
	Longitude float64 `json:"longitude,omitempty"`

	// Name: The name of the early vote site or drop off location. This
	// field is not populated for polling locations.
	Name string `json:"name,omitempty"`

	// Notes: Notes about this location (e.g. accessibility ramp or entrance
	// to use).
	Notes string `json:"notes,omitempty"`

	// PollingHours: A description of when this location is open.
	PollingHours string `json:"pollingHours,omitempty"`

	// Sources: A list of sources for this location. If multiple sources are
	// listed the data has been aggregated from those sources.
	Sources []*Source `json:"sources,omitempty"`

	// StartDate: The first date that this early vote site or drop off
	// location may be used. This field is not populated for polling
	// locations.
	StartDate string `json:"startDate,omitempty"`

	// VoterServices: The services provided by this early vote site or drop
	// off location. This field is not populated for polling locations.
	VoterServices string `json:"voterServices,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Address") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Address") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *PollingLocation) MarshalJSON() ([]byte, error) {
	type NoMethod PollingLocation
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

func (s *PollingLocation) UnmarshalJSON(data []byte) error {
	type NoMethod PollingLocation
	var s1 struct {
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

type Precinct struct {
	// AdministrationRegionId: ID of the AdministrationRegion message for
	// this precinct. Corresponds to LocalityId xml tag.
	AdministrationRegionId string `json:"administrationRegionId,omitempty"`

	// ContestId: ID(s) of the Contest message(s) for this precinct.
	ContestId []string `json:"contestId,omitempty"`

	// DatasetId: Required. Dataset ID. What datasets our Precincts come
	// from.
	DatasetId int64 `json:"datasetId,omitempty,string"`

	// EarlyVoteSiteId: ID(s) of the PollingLocation message(s) for this
	// precinct.
	EarlyVoteSiteId []string `json:"earlyVoteSiteId,omitempty"`

	// ElectoralDistrictId: ID(s) of the ElectoralDistrict message(s) for
	// this precinct.
	ElectoralDistrictId []string `json:"electoralDistrictId,omitempty"`

	// Id: Required. A unique identifier for this precinct.
	Id string `json:"id,omitempty"`

	// MailOnly: Specifies if the precinct runs mail-only elections.
	MailOnly bool `json:"mailOnly,omitempty"`

	// Name: Required. The name of the precinct.
	Name string `json:"name,omitempty"`

	// Number: The number of the precinct.
	Number string `json:"number,omitempty"`

	// OcdId: Encouraged. The OCD ID of the precinct
	OcdId []string `json:"ocdId,omitempty"`

	// PollingLocationId: ID(s) of the PollingLocation message(s) for this
	// precinct.
	PollingLocationId []string `json:"pollingLocationId,omitempty"`

	// SpatialBoundaryId: ID(s) of the SpatialBoundary message(s) for this
	// precinct. Used to specify a geometrical boundary of the precinct.
	SpatialBoundaryId []string `json:"spatialBoundaryId,omitempty"`

	// SplitName: If present, this proto corresponds to one portion of split
	// precinct. Other portions of this precinct are guaranteed to have the
	// same `name`. If not present, this proto represents a full precicnt.
	SplitName string `json:"splitName,omitempty"`

	// Ward: Specifies the ward the precinct is contained within.
	Ward string `json:"ward,omitempty"`

	// ForceSendFields is a list of field names (e.g.
	// "AdministrationRegionId") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. However, any non-pointer, non-interface field appearing in
	// ForceSendFields will be sent to the server regardless of whether the
	// field is empty or not. This may be used to include empty fields in
	// Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "AdministrationRegionId")
	// to include in API requests with the JSON null value. By default,
	// fields with empty values are omitted from API requests. However, any
	// field with an empty value appearing in NullFields will be sent to the
	// server as null. It is an error if a field in this list has a
	// non-empty value. This may be used to include null fields in Patch
	// requests.
	NullFields []string `json:"-"`
}

func (s *Precinct) MarshalJSON() ([]byte, error) {
	type NoMethod Precinct
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

type RepresentativeInfoData struct {
	// Divisions: A map of political geographic divisions that contain the
	// requested address, keyed by the unique Open Civic Data identifier for
	// this division.
	Divisions map[string]GeographicDivision `json:"divisions,omitempty"`

	// Offices: Elected offices referenced by the divisions listed above.
	// Will only be present if includeOffices was true in the request.
	Offices []*Office `json:"offices,omitempty"`

	// Officials: Officials holding the offices listed above. Will only be
	// present if includeOffices was true in the request.
	Officials []*Official `json:"officials,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Divisions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Divisions") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *RepresentativeInfoData) MarshalJSON() ([]byte, error) {
	type NoMethod RepresentativeInfoData
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// RepresentativeInfoResponse: The result of a representative info
// lookup query.
type RepresentativeInfoResponse struct {
	// Divisions: A map of political geographic divisions that contain the
	// requested address, keyed by the unique Open Civic Data identifier for
	// this division.
	Divisions map[string]GeographicDivision `json:"divisions,omitempty"`

	// Kind: Identifies what kind of resource this is. Value: the fixed
	// string "civicinfo#representativeInfoResponse".
	Kind string `json:"kind,omitempty"`

	// NormalizedInput: The normalized version of the requested address
	NormalizedInput *SimpleAddressType `json:"normalizedInput,omitempty"`

	// Offices: Elected offices referenced by the divisions listed above.
	// Will only be present if includeOffices was true in the request.
	Offices []*Office `json:"offices,omitempty"`

	// Officials: Officials holding the offices listed above. Will only be
	// present if includeOffices was true in the request.
	Officials []*Official `json:"officials,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Divisions") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Divisions") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *RepresentativeInfoResponse) MarshalJSON() ([]byte, error) {
	type NoMethod RepresentativeInfoResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// SimpleAddressType: A simple representation of an address.
type SimpleAddressType struct {
	// City: The city or town for the address.
	City string `json:"city,omitempty"`

	// Line1: The street name and number of this address.
	Line1 string `json:"line1,omitempty"`

	// Line2: The second line the address, if needed.
	Line2 string `json:"line2,omitempty"`

	// Line3: The third line of the address, if needed.
	Line3 string `json:"line3,omitempty"`

	// LocationName: The name of the location.
	LocationName string `json:"locationName,omitempty"`

	// State: The US two letter state abbreviation of the address.
	State string `json:"state,omitempty"`

	// Zip: The US Postal Zip Code of the address.
	Zip string `json:"zip,omitempty"`

	// ForceSendFields is a list of field names (e.g. "City") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "City") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *SimpleAddressType) MarshalJSON() ([]byte, error) {
	type NoMethod SimpleAddressType
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// Source: Contains information about the data source for the element
// containing it.
type Source struct {
	// Name: The name of the data source.
	Name string `json:"name,omitempty"`

	// Official: Whether this data comes from an official government source.
	Official bool `json:"official,omitempty"`

	// ForceSendFields is a list of field names (e.g. "Name") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Name") to include in API
	// requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *Source) MarshalJSON() ([]byte, error) {
	type NoMethod Source
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// VoterInfoResponse: The result of a voter info lookup query.
type VoterInfoResponse struct {
	// Contests: Contests that will appear on the voter's ballot.
	Contests []*Contest `json:"contests,omitempty"`

	// DropOffLocations: Locations where a voter is eligible to drop off a
	// completed ballot. The voter must have received and completed a ballot
	// prior to arriving at the location. The location may not have ballots
	// available on the premises. These locations could be open on or before
	// election day as indicated in the pollingHours field.
	DropOffLocations []*PollingLocation `json:"dropOffLocations,omitempty"`

	// EarlyVoteSites: Locations where the voter is eligible to vote early,
	// prior to election day.
	EarlyVoteSites []*PollingLocation `json:"earlyVoteSites,omitempty"`

	// Election: The election that was queried.
	Election *Election `json:"election,omitempty"`

	// Kind: Identifies what kind of resource this is. Value: the fixed
	// string "civicinfo#voterInfoResponse".
	Kind string `json:"kind,omitempty"`

	// MailOnly: Specifies whether voters in the precinct vote only by
	// mailing their ballots (with the possible option of dropping off their
	// ballots as well).
	MailOnly bool `json:"mailOnly,omitempty"`

	// NormalizedInput: The normalized version of the requested address
	NormalizedInput *SimpleAddressType `json:"normalizedInput,omitempty"`

	// OtherElections: When there are multiple elections for a voter
	// address, the otherElections field is populated in the API response
	// and there are two possibilities: 1. If the earliest election is not
	// the intended election, specify the election ID of the desired
	// election in a second API request using the electionId field. 2. If
	// these elections occur on the same day, the API doesn?t return any
	// polling location, contest, or election official information to ensure
	// that an additional query is made. For user-facing applications, we
	// recommend displaying these elections to the user to disambiguate. A
	// second API request using the electionId field should be made for the
	// election that is relevant to the user.
	OtherElections []*Election `json:"otherElections,omitempty"`

	// PollingLocations: Locations where the voter is eligible to vote on
	// election day.
	PollingLocations []*PollingLocation `json:"pollingLocations,omitempty"`

	PrecinctId string `json:"precinctId,omitempty"`

	// Precincts: The precincts that match this voter's address. Will only
	// be returned for project IDs which have been whitelisted as "partner
	// projects".
	Precincts []*Precinct `json:"precincts,omitempty"`

	// State: Local Election Information for the state that the voter votes
	// in. For the US, there will only be one element in this array.
	State []*AdministrationRegion `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the
	// server.
	googleapi.ServerResponse `json:"-"`

	// ForceSendFields is a list of field names (e.g. "Contests") to
	// unconditionally include in API requests. By default, fields with
	// empty or default values are omitted from API requests. However, any
	// non-pointer, non-interface field appearing in ForceSendFields will be
	// sent to the server regardless of whether the field is empty or not.
	// This may be used to include empty fields in Patch requests.
	ForceSendFields []string `json:"-"`

	// NullFields is a list of field names (e.g. "Contests") to include in
	// API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. However, any field with an
	// empty value appearing in NullFields will be sent to the server as
	// null. It is an error if a field in this list has a non-empty value.
	// This may be used to include null fields in Patch requests.
	NullFields []string `json:"-"`
}

func (s *VoterInfoResponse) MarshalJSON() ([]byte, error) {
	type NoMethod VoterInfoResponse
	raw := NoMethod(*s)
	return gensupport.MarshalJSON(raw, s.ForceSendFields, s.NullFields)
}

// method id "civicinfo.divisions.search":

type DivisionsSearchCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Search: Searches for political divisions by their natural name or OCD
// ID.
func (r *DivisionsService) Search() *DivisionsSearchCall {
	c := &DivisionsSearchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Query sets the optional parameter "query": The search query. Queries
// can cover any parts of a OCD ID or a human readable division name.
// All words given in the query are treated as required patterns. In
// addition to that, most query operators of the Apache Lucene library
// are supported. See
// http://lucene.apache.org/core/2_9_4/queryparsersyntax.html
func (c *DivisionsSearchCall) Query(query string) *DivisionsSearchCall {
	c.urlParams_.Set("query", query)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *DivisionsSearchCall) Fields(s ...googleapi.Field) *DivisionsSearchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *DivisionsSearchCall) IfNoneMatch(entityTag string) *DivisionsSearchCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *DivisionsSearchCall) Context(ctx context.Context) *DivisionsSearchCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *DivisionsSearchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DivisionsSearchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "civicinfo/v2/divisions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "civicinfo.divisions.search" call.
// Exactly one of *DivisionSearchResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *DivisionSearchResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *DivisionsSearchCall) Do(opts ...googleapi.CallOption) (*DivisionSearchResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DivisionSearchResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Searches for political divisions by their natural name or OCD ID.",
	//   "flatPath": "civicinfo/v2/divisions",
	//   "httpMethod": "GET",
	//   "id": "civicinfo.divisions.search",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "query": {
	//       "description": "The search query. Queries can cover any parts of a OCD ID or a human readable division name. All words given in the query are treated as required patterns. In addition to that, most query operators of the Apache Lucene library are supported. See http://lucene.apache.org/core/2_9_4/queryparsersyntax.html",
	//       "location": "query",
	//       "type": "string"
	//     }
	//   },
	//   "path": "civicinfo/v2/divisions",
	//   "response": {
	//     "$ref": "DivisionSearchResponse"
	//   }
	// }

}

// method id "civicinfo.elections.electionQuery":

type ElectionsElectionQueryCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ElectionQuery: List of available elections to query.
func (r *ElectionsService) ElectionQuery() *ElectionsElectionQueryCall {
	c := &ElectionsElectionQueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *ElectionsElectionQueryCall) Fields(s ...googleapi.Field) *ElectionsElectionQueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *ElectionsElectionQueryCall) IfNoneMatch(entityTag string) *ElectionsElectionQueryCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *ElectionsElectionQueryCall) Context(ctx context.Context) *ElectionsElectionQueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *ElectionsElectionQueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ElectionsElectionQueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "civicinfo/v2/elections")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "civicinfo.elections.electionQuery" call.
// Exactly one of *ElectionsQueryResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *ElectionsQueryResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *ElectionsElectionQueryCall) Do(opts ...googleapi.CallOption) (*ElectionsQueryResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ElectionsQueryResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "List of available elections to query.",
	//   "flatPath": "civicinfo/v2/elections",
	//   "httpMethod": "GET",
	//   "id": "civicinfo.elections.electionQuery",
	//   "parameterOrder": [],
	//   "parameters": {},
	//   "path": "civicinfo/v2/elections",
	//   "response": {
	//     "$ref": "ElectionsQueryResponse"
	//   }
	// }

}

// method id "civicinfo.elections.voterInfoQuery":

type ElectionsVoterInfoQueryCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// VoterInfoQuery: Looks up information relevant to a voter based on the
// voter's registered address.
//
// - address: The registered address of the voter to look up.
func (r *ElectionsService) VoterInfoQuery(address string) *ElectionsVoterInfoQueryCall {
	c := &ElectionsVoterInfoQueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("address", address)
	return c
}

// ElectionId sets the optional parameter "electionId": The unique ID of
// the election to look up. A list of election IDs can be obtained at
// https://www.googleapis.com/civicinfo/{version}/elections. If no
// election ID is specified in the query and there is more than one
// election with data for the given voter, the additional elections are
// provided in the otherElections response field.
func (c *ElectionsVoterInfoQueryCall) ElectionId(electionId int64) *ElectionsVoterInfoQueryCall {
	c.urlParams_.Set("electionId", fmt.Sprint(electionId))
	return c
}

// OfficialOnly sets the optional parameter "officialOnly": If set to
// true, only data from official state sources will be returned.
func (c *ElectionsVoterInfoQueryCall) OfficialOnly(officialOnly bool) *ElectionsVoterInfoQueryCall {
	c.urlParams_.Set("officialOnly", fmt.Sprint(officialOnly))
	return c
}

// ReturnAllAvailableData sets the optional parameter
// "returnAllAvailableData": If set to true, the query will return the
// success code and include any partial information when it is unable to
// determine a matching address or unable to determine the election for
// electionId=0 queries.
func (c *ElectionsVoterInfoQueryCall) ReturnAllAvailableData(returnAllAvailableData bool) *ElectionsVoterInfoQueryCall {
	c.urlParams_.Set("returnAllAvailableData", fmt.Sprint(returnAllAvailableData))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *ElectionsVoterInfoQueryCall) Fields(s ...googleapi.Field) *ElectionsVoterInfoQueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *ElectionsVoterInfoQueryCall) IfNoneMatch(entityTag string) *ElectionsVoterInfoQueryCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *ElectionsVoterInfoQueryCall) Context(ctx context.Context) *ElectionsVoterInfoQueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *ElectionsVoterInfoQueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ElectionsVoterInfoQueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "civicinfo/v2/voterinfo")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "civicinfo.elections.voterInfoQuery" call.
// Exactly one of *VoterInfoResponse or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *VoterInfoResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *ElectionsVoterInfoQueryCall) Do(opts ...googleapi.CallOption) (*VoterInfoResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &VoterInfoResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Looks up information relevant to a voter based on the voter's registered address.",
	//   "flatPath": "civicinfo/v2/voterinfo",
	//   "httpMethod": "GET",
	//   "id": "civicinfo.elections.voterInfoQuery",
	//   "parameterOrder": [
	//     "address"
	//   ],
	//   "parameters": {
	//     "address": {
	//       "description": "The registered address of the voter to look up.",
	//       "location": "query",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "electionId": {
	//       "default": "0",
	//       "description": "The unique ID of the election to look up. A list of election IDs can be obtained at https://www.googleapis.com/civicinfo/{version}/elections. If no election ID is specified in the query and there is more than one election with data for the given voter, the additional elections are provided in the otherElections response field.",
	//       "format": "int64",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "officialOnly": {
	//       "default": "false",
	//       "description": "If set to true, only data from official state sources will be returned.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "returnAllAvailableData": {
	//       "default": "false",
	//       "description": "If set to true, the query will return the success code and include any partial information when it is unable to determine a matching address or unable to determine the election for electionId=0 queries.",
	//       "location": "query",
	//       "type": "boolean"
	//     }
	//   },
	//   "path": "civicinfo/v2/voterinfo",
	//   "response": {
	//     "$ref": "VoterInfoResponse"
	//   }
	// }

}

// method id "civicinfo.representatives.representativeInfoByAddress":

type RepresentativesRepresentativeInfoByAddressCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// RepresentativeInfoByAddress: Looks up political geography and
// representative information for a single address.
func (r *RepresentativesService) RepresentativeInfoByAddress() *RepresentativesRepresentativeInfoByAddressCall {
	c := &RepresentativesRepresentativeInfoByAddressCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Address sets the optional parameter "address": The address to look
// up. May only be specified if the field ocdId is not given in the URL
func (c *RepresentativesRepresentativeInfoByAddressCall) Address(address string) *RepresentativesRepresentativeInfoByAddressCall {
	c.urlParams_.Set("address", address)
	return c
}

// IncludeOffices sets the optional parameter "includeOffices": Whether
// to return information about offices and officials. If false, only the
// top-level district information will be returned.
func (c *RepresentativesRepresentativeInfoByAddressCall) IncludeOffices(includeOffices bool) *RepresentativesRepresentativeInfoByAddressCall {
	c.urlParams_.Set("includeOffices", fmt.Sprint(includeOffices))
	return c
}

// Levels sets the optional parameter "levels": A list of office levels
// to filter by. Only offices that serve at least one of these levels
// will be returned. Divisions that don't contain a matching office will
// not be returned.
//
// Possible values:
//
//	"international"
//	"country"
//	"administrativeArea1"
//	"regional"
//	"administrativeArea2"
//	"locality"
//	"subLocality1"
//	"subLocality2"
//	"special"
func (c *RepresentativesRepresentativeInfoByAddressCall) Levels(levels ...string) *RepresentativesRepresentativeInfoByAddressCall {
	c.urlParams_.SetMulti("levels", append([]string{}, levels...))
	return c
}

// Roles sets the optional parameter "roles": A list of office roles to
// filter by. Only offices fulfilling one of these roles will be
// returned. Divisions that don't contain a matching office will not be
// returned.
//
// Possible values:
//
//	"headOfState"
//	"headOfGovernment"
//	"deputyHeadOfGovernment"
//	"governmentOfficer"
//	"executiveCouncil"
//	"legislatorUpperBody"
//	"legislatorLowerBody"
//	"highestCourtJudge"
//	"judge"
//	"schoolBoard"
//	"specialPurposeOfficer"
//	"otherRole"
func (c *RepresentativesRepresentativeInfoByAddressCall) Roles(roles ...string) *RepresentativesRepresentativeInfoByAddressCall {
	c.urlParams_.SetMulti("roles", append([]string{}, roles...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *RepresentativesRepresentativeInfoByAddressCall) Fields(s ...googleapi.Field) *RepresentativesRepresentativeInfoByAddressCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *RepresentativesRepresentativeInfoByAddressCall) IfNoneMatch(entityTag string) *RepresentativesRepresentativeInfoByAddressCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *RepresentativesRepresentativeInfoByAddressCall) Context(ctx context.Context) *RepresentativesRepresentativeInfoByAddressCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *RepresentativesRepresentativeInfoByAddressCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepresentativesRepresentativeInfoByAddressCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "civicinfo/v2/representatives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "civicinfo.representatives.representativeInfoByAddress" call.
// Exactly one of *RepresentativeInfoResponse or error will be non-nil.
// Any non-2xx status code is an error. Response headers are in either
// *RepresentativeInfoResponse.ServerResponse.Header or (if a response
// was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *RepresentativesRepresentativeInfoByAddressCall) Do(opts ...googleapi.CallOption) (*RepresentativeInfoResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RepresentativeInfoResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Looks up political geography and representative information for a single address.",
	//   "flatPath": "civicinfo/v2/representatives",
	//   "httpMethod": "GET",
	//   "id": "civicinfo.representatives.representativeInfoByAddress",
	//   "parameterOrder": [],
	//   "parameters": {
	//     "address": {
	//       "description": "The address to look up. May only be specified if the field ocdId is not given in the URL",
	//       "location": "query",
	//       "type": "string"
	//     },
	//     "includeOffices": {
	//       "default": "true",
	//       "description": "Whether to return information about offices and officials. If false, only the top-level district information will be returned.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "levels": {
	//       "description": "A list of office levels to filter by. Only offices that serve at least one of these levels will be returned. Divisions that don't contain a matching office will not be returned.",
	//       "enum": [
	//         "international",
	//         "country",
	//         "administrativeArea1",
	//         "regional",
	//         "administrativeArea2",
	//         "locality",
	//         "subLocality1",
	//         "subLocality2",
	//         "special"
	//       ],
	//       "enumDescriptions": [
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         ""
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "roles": {
	//       "description": "A list of office roles to filter by. Only offices fulfilling one of these roles will be returned. Divisions that don't contain a matching office will not be returned.",
	//       "enum": [
	//         "headOfState",
	//         "headOfGovernment",
	//         "deputyHeadOfGovernment",
	//         "governmentOfficer",
	//         "executiveCouncil",
	//         "legislatorUpperBody",
	//         "legislatorLowerBody",
	//         "highestCourtJudge",
	//         "judge",
	//         "schoolBoard",
	//         "specialPurposeOfficer",
	//         "otherRole"
	//       ],
	//       "enumDescriptions": [
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         ""
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "civicinfo/v2/representatives",
	//   "response": {
	//     "$ref": "RepresentativeInfoResponse"
	//   }
	// }

}

// method id "civicinfo.representatives.representativeInfoByDivision":

type RepresentativesRepresentativeInfoByDivisionCall struct {
	s            *Service
	ocdId        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// RepresentativeInfoByDivision: Looks up representative information for
// a single geographic division.
//
//   - ocdId: The Open Civic Data division identifier of the division to
//     look up.
func (r *RepresentativesService) RepresentativeInfoByDivision(ocdId string) *RepresentativesRepresentativeInfoByDivisionCall {
	c := &RepresentativesRepresentativeInfoByDivisionCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.ocdId = ocdId
	return c
}

// Levels sets the optional parameter "levels": A list of office levels
// to filter by. Only offices that serve at least one of these levels
// will be returned. Divisions that don't contain a matching office will
// not be returned.
//
// Possible values:
//
//	"international"
//	"country"
//	"administrativeArea1"
//	"regional"
//	"administrativeArea2"
//	"locality"
//	"subLocality1"
//	"subLocality2"
//	"special"
func (c *RepresentativesRepresentativeInfoByDivisionCall) Levels(levels ...string) *RepresentativesRepresentativeInfoByDivisionCall {
	c.urlParams_.SetMulti("levels", append([]string{}, levels...))
	return c
}

// Recursive sets the optional parameter "recursive": If true,
// information about all divisions contained in the division requested
// will be included as well. For example, if querying
// ocd-division/country:us/district:dc, this would also return all DC's
// wards and ANCs.
func (c *RepresentativesRepresentativeInfoByDivisionCall) Recursive(recursive bool) *RepresentativesRepresentativeInfoByDivisionCall {
	c.urlParams_.Set("recursive", fmt.Sprint(recursive))
	return c
}

// Roles sets the optional parameter "roles": A list of office roles to
// filter by. Only offices fulfilling one of these roles will be
// returned. Divisions that don't contain a matching office will not be
// returned.
//
// Possible values:
//
//	"headOfState"
//	"headOfGovernment"
//	"deputyHeadOfGovernment"
//	"governmentOfficer"
//	"executiveCouncil"
//	"legislatorUpperBody"
//	"legislatorLowerBody"
//	"highestCourtJudge"
//	"judge"
//	"schoolBoard"
//	"specialPurposeOfficer"
//	"otherRole"
func (c *RepresentativesRepresentativeInfoByDivisionCall) Roles(roles ...string) *RepresentativesRepresentativeInfoByDivisionCall {
	c.urlParams_.SetMulti("roles", append([]string{}, roles...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse
// for more information.
func (c *RepresentativesRepresentativeInfoByDivisionCall) Fields(s ...googleapi.Field) *RepresentativesRepresentativeInfoByDivisionCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets the optional parameter which makes the operation
// fail if the object's ETag matches the given value. This is useful for
// getting updates only after the object has changed since the last
// request. Use googleapi.IsNotModified to check whether the response
// error from Do is the result of In-None-Match.
func (c *RepresentativesRepresentativeInfoByDivisionCall) IfNoneMatch(entityTag string) *RepresentativesRepresentativeInfoByDivisionCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method. Any
// pending HTTP request will be aborted if the provided context is
// canceled.
func (c *RepresentativesRepresentativeInfoByDivisionCall) Context(ctx context.Context) *RepresentativesRepresentativeInfoByDivisionCall {
	c.ctx_ = ctx
	return c
}

// Header returns an http.Header that can be modified by the caller to
// add HTTP headers to the request.
func (c *RepresentativesRepresentativeInfoByDivisionCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepresentativesRepresentativeInfoByDivisionCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := make(http.Header)
	reqHeaders.Set("x-goog-api-client", "gl-go/"+gensupport.GoVersion()+" gdcl/"+internal.Version)
	for k, v := range c.header_ {
		reqHeaders[k] = v
	}
	reqHeaders.Set("User-Agent", c.s.userAgent())
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "civicinfo/v2/representatives/{ocdId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"ocdId": c.ocdId,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "civicinfo.representatives.representativeInfoByDivision" call.
// Exactly one of *RepresentativeInfoData or error will be non-nil. Any
// non-2xx status code is an error. Response headers are in either
// *RepresentativeInfoData.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was
// because http.StatusNotModified was returned.
func (c *RepresentativesRepresentativeInfoByDivisionCall) Do(opts ...googleapi.CallOption) (*RepresentativeInfoData, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RepresentativeInfoData{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
	// {
	//   "description": "Looks up representative information for a single geographic division.",
	//   "flatPath": "civicinfo/v2/representatives/{ocdId}",
	//   "httpMethod": "GET",
	//   "id": "civicinfo.representatives.representativeInfoByDivision",
	//   "parameterOrder": [
	//     "ocdId"
	//   ],
	//   "parameters": {
	//     "levels": {
	//       "description": "A list of office levels to filter by. Only offices that serve at least one of these levels will be returned. Divisions that don't contain a matching office will not be returned.",
	//       "enum": [
	//         "international",
	//         "country",
	//         "administrativeArea1",
	//         "regional",
	//         "administrativeArea2",
	//         "locality",
	//         "subLocality1",
	//         "subLocality2",
	//         "special"
	//       ],
	//       "enumDescriptions": [
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         ""
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     },
	//     "ocdId": {
	//       "description": "The Open Civic Data division identifier of the division to look up.",
	//       "location": "path",
	//       "required": true,
	//       "type": "string"
	//     },
	//     "recursive": {
	//       "description": "If true, information about all divisions contained in the division requested will be included as well. For example, if querying ocd-division/country:us/district:dc, this would also return all DC's wards and ANCs.",
	//       "location": "query",
	//       "type": "boolean"
	//     },
	//     "roles": {
	//       "description": "A list of office roles to filter by. Only offices fulfilling one of these roles will be returned. Divisions that don't contain a matching office will not be returned.",
	//       "enum": [
	//         "headOfState",
	//         "headOfGovernment",
	//         "deputyHeadOfGovernment",
	//         "governmentOfficer",
	//         "executiveCouncil",
	//         "legislatorUpperBody",
	//         "legislatorLowerBody",
	//         "highestCourtJudge",
	//         "judge",
	//         "schoolBoard",
	//         "specialPurposeOfficer",
	//         "otherRole"
	//       ],
	//       "enumDescriptions": [
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         "",
	//         ""
	//       ],
	//       "location": "query",
	//       "repeated": true,
	//       "type": "string"
	//     }
	//   },
	//   "path": "civicinfo/v2/representatives/{ocdId}",
	//   "response": {
	//     "$ref": "RepresentativeInfoData"
	//   }
	// }

}
