// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package monitoring provides access to the Cloud Monitoring API.
//
// For product documentation, see: https://cloud.google.com/monitoring/api/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/monitoring/v1"
//	...
//	ctx := context.Background()
//	monitoringService, err := monitoring.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	monitoringService, err := monitoring.NewService(ctx, option.WithScopes(monitoring.MonitoringWriteScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	monitoringService, err := monitoring.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	monitoringService, err := monitoring.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package monitoring // import "google.golang.org/api/monitoring/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "monitoring:v1"
const apiName = "monitoring"
const apiVersion = "v1"
const basePath = "https://monitoring.googleapis.com/"
const basePathTemplate = "https://monitoring.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://monitoring.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"

	// View and write monitoring data for all of your Google and third-party Cloud
	// and API projects
	MonitoringScope = "https://www.googleapis.com/auth/monitoring"

	// View monitoring data for all of your Google Cloud and third-party projects
	MonitoringReadScope = "https://www.googleapis.com/auth/monitoring.read"

	// Publish metric data to your Google Cloud projects
	MonitoringWriteScope = "https://www.googleapis.com/auth/monitoring.write"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
		"https://www.googleapis.com/auth/monitoring",
		"https://www.googleapis.com/auth/monitoring.read",
		"https://www.googleapis.com/auth/monitoring.write",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Locations = NewLocationsService(s)
	s.Operations = NewOperationsService(s)
	s.Projects = NewProjectsService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Locations *LocationsService

	Operations *OperationsService

	Projects *ProjectsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewLocationsService(s *Service) *LocationsService {
	rs := &LocationsService{s: s}
	rs.Global = NewLocationsGlobalService(s)
	return rs
}

type LocationsService struct {
	s *Service

	Global *LocationsGlobalService
}

func NewLocationsGlobalService(s *Service) *LocationsGlobalService {
	rs := &LocationsGlobalService{s: s}
	rs.MetricsScopes = NewLocationsGlobalMetricsScopesService(s)
	return rs
}

type LocationsGlobalService struct {
	s *Service

	MetricsScopes *LocationsGlobalMetricsScopesService
}

func NewLocationsGlobalMetricsScopesService(s *Service) *LocationsGlobalMetricsScopesService {
	rs := &LocationsGlobalMetricsScopesService{s: s}
	rs.Projects = NewLocationsGlobalMetricsScopesProjectsService(s)
	return rs
}

type LocationsGlobalMetricsScopesService struct {
	s *Service

	Projects *LocationsGlobalMetricsScopesProjectsService
}

func NewLocationsGlobalMetricsScopesProjectsService(s *Service) *LocationsGlobalMetricsScopesProjectsService {
	rs := &LocationsGlobalMetricsScopesProjectsService{s: s}
	return rs
}

type LocationsGlobalMetricsScopesProjectsService struct {
	s *Service
}

func NewOperationsService(s *Service) *OperationsService {
	rs := &OperationsService{s: s}
	return rs
}

type OperationsService struct {
	s *Service
}

func NewProjectsService(s *Service) *ProjectsService {
	rs := &ProjectsService{s: s}
	rs.Dashboards = NewProjectsDashboardsService(s)
	rs.Location = NewProjectsLocationService(s)
	return rs
}

type ProjectsService struct {
	s *Service

	Dashboards *ProjectsDashboardsService

	Location *ProjectsLocationService
}

func NewProjectsDashboardsService(s *Service) *ProjectsDashboardsService {
	rs := &ProjectsDashboardsService{s: s}
	return rs
}

type ProjectsDashboardsService struct {
	s *Service
}

func NewProjectsLocationService(s *Service) *ProjectsLocationService {
	rs := &ProjectsLocationService{s: s}
	rs.Prometheus = NewProjectsLocationPrometheusService(s)
	return rs
}

type ProjectsLocationService struct {
	s *Service

	Prometheus *ProjectsLocationPrometheusService
}

func NewProjectsLocationPrometheusService(s *Service) *ProjectsLocationPrometheusService {
	rs := &ProjectsLocationPrometheusService{s: s}
	rs.Api = NewProjectsLocationPrometheusApiService(s)
	return rs
}

type ProjectsLocationPrometheusService struct {
	s *Service

	Api *ProjectsLocationPrometheusApiService
}

func NewProjectsLocationPrometheusApiService(s *Service) *ProjectsLocationPrometheusApiService {
	rs := &ProjectsLocationPrometheusApiService{s: s}
	rs.V1 = NewProjectsLocationPrometheusApiV1Service(s)
	return rs
}

type ProjectsLocationPrometheusApiService struct {
	s *Service

	V1 *ProjectsLocationPrometheusApiV1Service
}

func NewProjectsLocationPrometheusApiV1Service(s *Service) *ProjectsLocationPrometheusApiV1Service {
	rs := &ProjectsLocationPrometheusApiV1Service{s: s}
	rs.Label = NewProjectsLocationPrometheusApiV1LabelService(s)
	rs.Metadata = NewProjectsLocationPrometheusApiV1MetadataService(s)
	return rs
}

type ProjectsLocationPrometheusApiV1Service struct {
	s *Service

	Label *ProjectsLocationPrometheusApiV1LabelService

	Metadata *ProjectsLocationPrometheusApiV1MetadataService
}

func NewProjectsLocationPrometheusApiV1LabelService(s *Service) *ProjectsLocationPrometheusApiV1LabelService {
	rs := &ProjectsLocationPrometheusApiV1LabelService{s: s}
	return rs
}

type ProjectsLocationPrometheusApiV1LabelService struct {
	s *Service
}

func NewProjectsLocationPrometheusApiV1MetadataService(s *Service) *ProjectsLocationPrometheusApiV1MetadataService {
	rs := &ProjectsLocationPrometheusApiV1MetadataService{s: s}
	return rs
}

type ProjectsLocationPrometheusApiV1MetadataService struct {
	s *Service
}

// Aggregation: Describes how to combine multiple time series to provide a
// different view of the data. Aggregation of time series is done in two steps.
// First, each time series in the set is aligned to the same time interval
// boundaries, then the set of time series is optionally reduced in
// number.Alignment consists of applying the per_series_aligner operation to
// each time series after its data has been divided into regular
// alignment_period time intervals. This process takes all of the data points
// in an alignment period, applies a mathematical transformation such as
// averaging, minimum, maximum, delta, etc., and converts them into a single
// data point per period.Reduction is when the aligned and transformed time
// series can optionally be combined, reducing the number of time series
// through similar mathematical transformations. Reduction involves applying a
// cross_series_reducer to all the time series, optionally sorting the time
// series into subsets with group_by_fields, and applying the reducer to each
// subset.The raw time series data can contain a huge amount of information
// from multiple sources. Alignment and reduction transforms this mass of data
// into a more manageable and representative collection of data, for example
// "the 95% latency across the average of all tasks in a cluster". This
// representative data can be more easily graphed and comprehended, and the
// individual time series data is still available for later drilldown. For more
// details, see Filtering and aggregation
// (https://cloud.google.com/monitoring/api/v3/aggregation).
type Aggregation struct {
	// AlignmentPeriod: The alignment_period specifies a time interval, in seconds,
	// that is used to divide the data in all the time series into consistent
	// blocks of time. This will be done before the per-series aligner can be
	// applied to the data.The value must be at least 60 seconds. If a per-series
	// aligner other than ALIGN_NONE is specified, this field is required or an
	// error is returned. If no per-series aligner is specified, or the aligner
	// ALIGN_NONE is specified, then this field is ignored.The maximum value of the
	// alignment_period is 2 years, or 104 weeks.
	AlignmentPeriod string `json:"alignmentPeriod,omitempty"`
	// CrossSeriesReducer: The reduction operation to be used to combine time
	// series into a single time series, where the value of each data point in the
	// resulting series is a function of all the already aligned values in the
	// input time series.Not all reducer operations can be applied to all time
	// series. The valid choices depend on the metric_kind and the value_type of
	// the original time series. Reduction can yield a time series with a different
	// metric_kind or value_type than the input time series.Time series data must
	// first be aligned (see per_series_aligner) in order to perform cross-time
	// series reduction. If cross_series_reducer is specified, then
	// per_series_aligner must be specified, and must not be ALIGN_NONE. An
	// alignment_period must also be specified; otherwise, an error is returned.
	//
	// Possible values:
	//   "REDUCE_NONE" - No cross-time series reduction. The output of the Aligner
	// is returned.
	//   "REDUCE_MEAN" - Reduce by computing the mean value across time series for
	// each alignment period. This reducer is valid for DELTA and GAUGE metrics
	// with numeric or distribution values. The value_type of the output is DOUBLE.
	//   "REDUCE_MIN" - Reduce by computing the minimum value across time series
	// for each alignment period. This reducer is valid for DELTA and GAUGE metrics
	// with numeric values. The value_type of the output is the same as the
	// value_type of the input.
	//   "REDUCE_MAX" - Reduce by computing the maximum value across time series
	// for each alignment period. This reducer is valid for DELTA and GAUGE metrics
	// with numeric values. The value_type of the output is the same as the
	// value_type of the input.
	//   "REDUCE_SUM" - Reduce by computing the sum across time series for each
	// alignment period. This reducer is valid for DELTA and GAUGE metrics with
	// numeric and distribution values. The value_type of the output is the same as
	// the value_type of the input.
	//   "REDUCE_STDDEV" - Reduce by computing the standard deviation across time
	// series for each alignment period. This reducer is valid for DELTA and GAUGE
	// metrics with numeric or distribution values. The value_type of the output is
	// DOUBLE.
	//   "REDUCE_COUNT" - Reduce by computing the number of data points across time
	// series for each alignment period. This reducer is valid for DELTA and GAUGE
	// metrics of numeric, Boolean, distribution, and string value_type. The
	// value_type of the output is INT64.
	//   "REDUCE_COUNT_TRUE" - Reduce by computing the number of True-valued data
	// points across time series for each alignment period. This reducer is valid
	// for DELTA and GAUGE metrics of Boolean value_type. The value_type of the
	// output is INT64.
	//   "REDUCE_COUNT_FALSE" - Reduce by computing the number of False-valued data
	// points across time series for each alignment period. This reducer is valid
	// for DELTA and GAUGE metrics of Boolean value_type. The value_type of the
	// output is INT64.
	//   "REDUCE_FRACTION_TRUE" - Reduce by computing the ratio of the number of
	// True-valued data points to the total number of data points for each
	// alignment period. This reducer is valid for DELTA and GAUGE metrics of
	// Boolean value_type. The output value is in the range 0.0, 1.0 and has
	// value_type DOUBLE.
	//   "REDUCE_PERCENTILE_99" - Reduce by computing the 99th percentile
	// (https://en.wikipedia.org/wiki/Percentile) of data points across time series
	// for each alignment period. This reducer is valid for GAUGE and DELTA metrics
	// of numeric and distribution type. The value of the output is DOUBLE.
	//   "REDUCE_PERCENTILE_95" - Reduce by computing the 95th percentile
	// (https://en.wikipedia.org/wiki/Percentile) of data points across time series
	// for each alignment period. This reducer is valid for GAUGE and DELTA metrics
	// of numeric and distribution type. The value of the output is DOUBLE.
	//   "REDUCE_PERCENTILE_50" - Reduce by computing the 50th percentile
	// (https://en.wikipedia.org/wiki/Percentile) of data points across time series
	// for each alignment period. This reducer is valid for GAUGE and DELTA metrics
	// of numeric and distribution type. The value of the output is DOUBLE.
	//   "REDUCE_PERCENTILE_05" - Reduce by computing the 5th percentile
	// (https://en.wikipedia.org/wiki/Percentile) of data points across time series
	// for each alignment period. This reducer is valid for GAUGE and DELTA metrics
	// of numeric and distribution type. The value of the output is DOUBLE.
	CrossSeriesReducer string `json:"crossSeriesReducer,omitempty"`
	// GroupByFields: The set of fields to preserve when cross_series_reducer is
	// specified. The group_by_fields determine how the time series are partitioned
	// into subsets prior to applying the aggregation operation. Each subset
	// contains time series that have the same value for each of the grouping
	// fields. Each individual time series is a member of exactly one subset. The
	// cross_series_reducer is applied to each subset of time series. It is not
	// possible to reduce across different resource types, so this field implicitly
	// contains resource.type. Fields not specified in group_by_fields are
	// aggregated away. If group_by_fields is not specified and all the time series
	// have the same resource type, then the time series are aggregated into a
	// single output time series. If cross_series_reducer is not defined, this
	// field is ignored.
	GroupByFields []string `json:"groupByFields,omitempty"`
	// PerSeriesAligner: An Aligner describes how to bring the data points in a
	// single time series into temporal alignment. Except for ALIGN_NONE, all
	// alignments cause all the data points in an alignment_period to be
	// mathematically grouped together, resulting in a single data point for each
	// alignment_period with end timestamp at the end of the period.Not all
	// alignment operations may be applied to all time series. The valid choices
	// depend on the metric_kind and value_type of the original time series.
	// Alignment can change the metric_kind or the value_type of the time
	// series.Time series data must be aligned in order to perform cross-time
	// series reduction. If cross_series_reducer is specified, then
	// per_series_aligner must be specified and not equal to ALIGN_NONE and
	// alignment_period must be specified; otherwise, an error is returned.
	//
	// Possible values:
	//   "ALIGN_NONE" - No alignment. Raw data is returned. Not valid if
	// cross-series reduction is requested. The value_type of the result is the
	// same as the value_type of the input.
	//   "ALIGN_DELTA" - Align and convert to DELTA. The output is delta = y1 -
	// y0.This alignment is valid for CUMULATIVE and DELTA metrics. If the selected
	// alignment period results in periods with no data, then the aligned value for
	// such a period is created by interpolation. The value_type of the aligned
	// result is the same as the value_type of the input.
	//   "ALIGN_RATE" - Align and convert to a rate. The result is computed as rate
	// = (y1 - y0)/(t1 - t0), or "delta over time". Think of this aligner as
	// providing the slope of the line that passes through the value at the start
	// and at the end of the alignment_period.This aligner is valid for CUMULATIVE
	// and DELTA metrics with numeric values. If the selected alignment period
	// results in periods with no data, then the aligned value for such a period is
	// created by interpolation. The output is a GAUGE metric with value_type
	// DOUBLE.If, by "rate", you mean "percentage change", see the
	// ALIGN_PERCENT_CHANGE aligner instead.
	//   "ALIGN_INTERPOLATE" - Align by interpolating between adjacent points
	// around the alignment period boundary. This aligner is valid for GAUGE
	// metrics with numeric values. The value_type of the aligned result is the
	// same as the value_type of the input.
	//   "ALIGN_NEXT_OLDER" - Align by moving the most recent data point before the
	// end of the alignment period to the boundary at the end of the alignment
	// period. This aligner is valid for GAUGE metrics. The value_type of the
	// aligned result is the same as the value_type of the input.
	//   "ALIGN_MIN" - Align the time series by returning the minimum value in each
	// alignment period. This aligner is valid for GAUGE and DELTA metrics with
	// numeric values. The value_type of the aligned result is the same as the
	// value_type of the input.
	//   "ALIGN_MAX" - Align the time series by returning the maximum value in each
	// alignment period. This aligner is valid for GAUGE and DELTA metrics with
	// numeric values. The value_type of the aligned result is the same as the
	// value_type of the input.
	//   "ALIGN_MEAN" - Align the time series by returning the mean value in each
	// alignment period. This aligner is valid for GAUGE and DELTA metrics with
	// numeric values. The value_type of the aligned result is DOUBLE.
	//   "ALIGN_COUNT" - Align the time series by returning the number of values in
	// each alignment period. This aligner is valid for GAUGE and DELTA metrics
	// with numeric or Boolean values. The value_type of the aligned result is
	// INT64.
	//   "ALIGN_SUM" - Align the time series by returning the sum of the values in
	// each alignment period. This aligner is valid for GAUGE and DELTA metrics
	// with numeric and distribution values. The value_type of the aligned result
	// is the same as the value_type of the input.
	//   "ALIGN_STDDEV" - Align the time series by returning the standard deviation
	// of the values in each alignment period. This aligner is valid for GAUGE and
	// DELTA metrics with numeric values. The value_type of the output is DOUBLE.
	//   "ALIGN_COUNT_TRUE" - Align the time series by returning the number of True
	// values in each alignment period. This aligner is valid for GAUGE metrics
	// with Boolean values. The value_type of the output is INT64.
	//   "ALIGN_COUNT_FALSE" - Align the time series by returning the number of
	// False values in each alignment period. This aligner is valid for GAUGE
	// metrics with Boolean values. The value_type of the output is INT64.
	//   "ALIGN_FRACTION_TRUE" - Align the time series by returning the ratio of
	// the number of True values to the total number of values in each alignment
	// period. This aligner is valid for GAUGE metrics with Boolean values. The
	// output value is in the range 0.0, 1.0 and has value_type DOUBLE.
	//   "ALIGN_PERCENTILE_99" - Align the time series by using percentile
	// aggregation (https://en.wikipedia.org/wiki/Percentile). The resulting data
	// point in each alignment period is the 99th percentile of all data points in
	// the period. This aligner is valid for GAUGE and DELTA metrics with
	// distribution values. The output is a GAUGE metric with value_type DOUBLE.
	//   "ALIGN_PERCENTILE_95" - Align the time series by using percentile
	// aggregation (https://en.wikipedia.org/wiki/Percentile). The resulting data
	// point in each alignment period is the 95th percentile of all data points in
	// the period. This aligner is valid for GAUGE and DELTA metrics with
	// distribution values. The output is a GAUGE metric with value_type DOUBLE.
	//   "ALIGN_PERCENTILE_50" - Align the time series by using percentile
	// aggregation (https://en.wikipedia.org/wiki/Percentile). The resulting data
	// point in each alignment period is the 50th percentile of all data points in
	// the period. This aligner is valid for GAUGE and DELTA metrics with
	// distribution values. The output is a GAUGE metric with value_type DOUBLE.
	//   "ALIGN_PERCENTILE_05" - Align the time series by using percentile
	// aggregation (https://en.wikipedia.org/wiki/Percentile). The resulting data
	// point in each alignment period is the 5th percentile of all data points in
	// the period. This aligner is valid for GAUGE and DELTA metrics with
	// distribution values. The output is a GAUGE metric with value_type DOUBLE.
	//   "ALIGN_PERCENT_CHANGE" - Align and convert to a percentage change. This
	// aligner is valid for GAUGE and DELTA metrics with numeric values. This
	// alignment returns ((current - previous)/previous) * 100, where the value of
	// previous is determined based on the alignment_period.If the values of
	// current and previous are both 0, then the returned value is 0. If only
	// previous is 0, the returned value is infinity.A 10-minute moving mean is
	// computed at each point of the alignment period prior to the above
	// calculation to smooth the metric and prevent false positives from very
	// short-lived spikes. The moving mean is only applicable for data whose values
	// are >= 0. Any values < 0 are treated as a missing datapoint, and are
	// ignored. While DELTA metrics are accepted by this alignment, special care
	// should be taken that the values for the metric will always be positive. The
	// output is a GAUGE metric with value_type DOUBLE.
	PerSeriesAligner string `json:"perSeriesAligner,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AlignmentPeriod") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AlignmentPeriod") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Aggregation) MarshalJSON() ([]byte, error) {
	type NoMethod Aggregation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AggregationFunction: Preview: An identifier for an aggregation function.
// Aggregation functions are SQL functions that group or transform data from
// multiple points to a single point. This is a preview feature and may be
// subject to change before final release.
type AggregationFunction struct {
	// Parameters: Optional. Parameters applied to the aggregation function. Only
	// used for functions that require them.
	Parameters []*Parameter `json:"parameters,omitempty"`
	// Type: Required. The type of aggregation function, must be one of the
	// following: "none" - no function. "percentile" - APPROX_QUANTILES() - 1
	// parameter numeric value "average" - AVG() "count" - COUNT() "count-distinct"
	// - COUNT(DISTINCT) "count-distinct-approx" - APPROX_COUNT_DISTINCT() "max" -
	// MAX() "min" - MIN() "sum" - SUM()
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Parameters") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Parameters") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AggregationFunction) MarshalJSON() ([]byte, error) {
	type NoMethod AggregationFunction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AlertChart: A chart that displays alert policy data.
type AlertChart struct {
	// Name: Required. The resource name of the alert policy. The format is:
	// projects/[PROJECT_ID_OR_NUMBER]/alertPolicies/[ALERT_POLICY_ID]
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AlertChart) MarshalJSON() ([]byte, error) {
	type NoMethod AlertChart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Axis: A chart axis.
type Axis struct {
	// Label: The label of the axis.
	Label string `json:"label,omitempty"`
	// Scale: The axis scale. By default, a linear scale is used.
	//
	// Possible values:
	//   "SCALE_UNSPECIFIED" - Scale is unspecified. The view will default to
	// LINEAR.
	//   "LINEAR" - Linear scale.
	//   "LOG10" - Logarithmic scale (base 10).
	Scale string `json:"scale,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Label") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Label") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Axis) MarshalJSON() ([]byte, error) {
	type NoMethod Axis
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Breakdown: Preview: A breakdown is an aggregation applied to the measures
// over a specified column. A breakdown can result in multiple series across a
// category for the provided measure. This is a preview feature and may be
// subject to change before final release.
type Breakdown struct {
	// AggregationFunction: Required. The Aggregation function is applied across
	// all data in each breakdown created.
	AggregationFunction *AggregationFunction `json:"aggregationFunction,omitempty"`
	// Column: Required. The name of the column in the dataset containing the
	// breakdown values.
	Column string `json:"column,omitempty"`
	// Limit: Required. A limit to the number of breakdowns. If set to zero then
	// all possible breakdowns are applied. The list of breakdowns is dependent on
	// the value of the sort_order field.
	Limit int64 `json:"limit,omitempty"`
	// SortOrder: Required. The sort order is applied to the values of the
	// breakdown column.
	//
	// Possible values:
	//   "SORT_ORDER_UNSPECIFIED" - An unspecified sort order. This option is
	// invalid when sorting is required.
	//   "SORT_ORDER_NONE" - No sorting is applied.
	//   "SORT_ORDER_ASCENDING" - The lowest-valued entries are selected first.
	//   "SORT_ORDER_DESCENDING" - The highest-valued entries are selected first.
	SortOrder string `json:"sortOrder,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AggregationFunction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AggregationFunction") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Breakdown) MarshalJSON() ([]byte, error) {
	type NoMethod Breakdown
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChartOptions: Options to control visual rendering of a chart.
type ChartOptions struct {
	// DisplayHorizontal: Preview: Configures whether the charted values are shown
	// on the horizontal or vertical axis. By default, values are represented the
	// vertical axis. This is a preview feature and may be subject to change before
	// final release.
	DisplayHorizontal bool `json:"displayHorizontal,omitempty"`
	// Mode: The chart mode.
	//
	// Possible values:
	//   "MODE_UNSPECIFIED" - Mode is unspecified. The view will default to COLOR.
	//   "COLOR" - The chart distinguishes data series using different color. Line
	// colors may get reused when there are many lines in the chart.
	//   "X_RAY" - The chart uses the Stackdriver x-ray mode, in which each data
	// set is plotted using the same semi-transparent color.
	//   "STATS" - The chart displays statistics such as average, median, 95th
	// percentile, and more.
	Mode string `json:"mode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayHorizontal") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayHorizontal") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChartOptions) MarshalJSON() ([]byte, error) {
	type NoMethod ChartOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CollapsibleGroup: A widget that groups the other widgets. All widgets that
// are within the area spanned by the grouping widget are considered member
// widgets.
type CollapsibleGroup struct {
	// Collapsed: The collapsed state of the widget on first page load.
	Collapsed bool `json:"collapsed,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Collapsed") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Collapsed") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CollapsibleGroup) MarshalJSON() ([]byte, error) {
	type NoMethod CollapsibleGroup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Column: Defines the layout properties and content for a column.
type Column struct {
	// Weight: The relative weight of this column. The column weight is used to
	// adjust the width of columns on the screen (relative to peers). Greater the
	// weight, greater the width of the column on the screen. If omitted, a value
	// of 1 is used while rendering.
	Weight int64 `json:"weight,omitempty,string"`
	// Widgets: The display widgets arranged vertically in this column.
	Widgets []*Widget `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Weight") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Weight") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Column) MarshalJSON() ([]byte, error) {
	type NoMethod Column
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ColumnLayout: A simplified layout that divides the available space into
// vertical columns and arranges a set of widgets vertically in each column.
type ColumnLayout struct {
	// Columns: The columns of content to display.
	Columns []*Column `json:"columns,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Columns") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Columns") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ColumnLayout) MarshalJSON() ([]byte, error) {
	type NoMethod ColumnLayout
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ColumnSettings: The persistent settings for a table's columns.
type ColumnSettings struct {
	// Alignment: Optional. Whether the column should be left / middle / right
	// aligned
	//
	// Possible values:
	//   "CELL_ALIGNMENT_UNSPECIFIED" - No horizontal alignment specified; fall
	// back to the default behavior. Label values are left aligned. Numeric values
	// are right aligned.
	//   "LEFT" - Left-align
	//   "CENTER" - Center-align
	//   "RIGHT" - Right-align
	Alignment string `json:"alignment,omitempty"`
	// Column: Required. The id of the column.
	Column string `json:"column,omitempty"`
	// DisplayName: Optional. Display name of the column
	DisplayName string `json:"displayName,omitempty"`
	// Thresholds: Optional. The thresholds used to determine how the table cell
	// should be rendered given the time series' current value.
	Thresholds []*Threshold `json:"thresholds,omitempty"`
	// Visible: Required. Whether the column should be visible on page load.
	Visible bool `json:"visible,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Alignment") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Alignment") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ColumnSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ColumnSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ColumnSortingOptions: Data structure to storing column's sort strategy
type ColumnSortingOptions struct {
	// Column: Optional. Column name to sort data by
	Column string `json:"column,omitempty"`
	// Direction: Optional. A sorting direction that determines ascending or
	// descending order. This is a legacy field kept for backwards compatibility
	// with table.
	//
	// Possible values:
	//   "SORT_ORDER_UNSPECIFIED" - An unspecified sort order. This option is
	// invalid when sorting is required.
	//   "SORT_ORDER_NONE" - No sorting is applied.
	//   "SORT_ORDER_ASCENDING" - The lowest-valued entries are selected first.
	//   "SORT_ORDER_DESCENDING" - The highest-valued entries are selected first.
	Direction string `json:"direction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Column") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Column") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ColumnSortingOptions) MarshalJSON() ([]byte, error) {
	type NoMethod ColumnSortingOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Dashboard: A Google Stackdriver dashboard. Dashboards define the content and
// layout of pages in the Stackdriver web application.
type Dashboard struct {
	// Annotations: Configuration for event annotations to display on this
	// dashboard.
	Annotations *DashboardAnnotations `json:"annotations,omitempty"`
	// ColumnLayout: The content is divided into equally spaced columns and the
	// widgets are arranged vertically.
	ColumnLayout *ColumnLayout `json:"columnLayout,omitempty"`
	// DashboardFilters: Filters to reduce the amount of data charted based on the
	// filter criteria.
	DashboardFilters []*DashboardFilter `json:"dashboardFilters,omitempty"`
	// DisplayName: Required. The mutable, human-readable name.
	DisplayName string `json:"displayName,omitempty"`
	// Etag: etag is used for optimistic concurrency control as a way to help
	// prevent simultaneous updates of a policy from overwriting each other. An
	// etag is returned in the response to GetDashboard, and users are expected to
	// put that etag in the request to UpdateDashboard to ensure that their change
	// will be applied to the same version of the Dashboard configuration. The
	// field should not be passed during dashboard creation.
	Etag string `json:"etag,omitempty"`
	// GridLayout: Content is arranged with a basic layout that re-flows a simple
	// list of informational elements like widgets or tiles.
	GridLayout *GridLayout `json:"gridLayout,omitempty"`
	// Labels: Labels applied to the dashboard
	Labels map[string]string `json:"labels,omitempty"`
	// MosaicLayout: The content is arranged as a grid of tiles, with each content
	// widget occupying one or more grid blocks.
	MosaicLayout *MosaicLayout `json:"mosaicLayout,omitempty"`
	// Name: Identifier. The resource name of the dashboard.
	Name string `json:"name,omitempty"`
	// RowLayout: The content is divided into equally spaced rows and the widgets
	// are arranged horizontally.
	RowLayout *RowLayout `json:"rowLayout,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Annotations") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Annotations") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Dashboard) MarshalJSON() ([]byte, error) {
	type NoMethod Dashboard
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DashboardAnnotations: Dashboard-level configuration for annotations
type DashboardAnnotations struct {
	// DefaultResourceNames: Dashboard level defaults for names of logging
	// resources to search for events. Currently only projects are supported. Each
	// individual EventAnnotation may have its own overrides. If both this field
	// and the per annotation field is empty, then the scoping project is used.
	// Limit: 50 projects. For example: “projects/some-project-id”
	DefaultResourceNames []string `json:"defaultResourceNames,omitempty"`
	// EventAnnotations: List of annotation configurations for this dashboard. Each
	// entry specifies one event type.
	EventAnnotations []*EventAnnotation `json:"eventAnnotations,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DefaultResourceNames") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DefaultResourceNames") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DashboardAnnotations) MarshalJSON() ([]byte, error) {
	type NoMethod DashboardAnnotations
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DashboardFilter: A filter to reduce the amount of data charted in relevant
// widgets.
type DashboardFilter struct {
	// FilterType: The specified filter type
	//
	// Possible values:
	//   "FILTER_TYPE_UNSPECIFIED" - Filter type is unspecified. This is not valid
	// in a well-formed request.
	//   "RESOURCE_LABEL" - Filter on a resource label value
	//   "METRIC_LABEL" - Filter on a metrics label value
	//   "USER_METADATA_LABEL" - Filter on a user metadata label value
	//   "SYSTEM_METADATA_LABEL" - Filter on a system metadata label value
	//   "GROUP" - Filter on a group id
	//   "VALUE_ONLY" - Filter that only contains a value. The label_key field must
	// be unset for filters of this type.
	FilterType string `json:"filterType,omitempty"`
	// LabelKey: Optional. The key for the label. This must be omitted if the
	// filter_type is VALUE_ONLY but is required otherwise.
	LabelKey string `json:"labelKey,omitempty"`
	// StringArray: A list of possible string values for the filter
	StringArray *StringArray `json:"stringArray,omitempty"`
	// StringArrayValue: An array of variable-length string values. If this field
	// is set, value_type must be set to STRING_ARRAY or VALUE_TYPE_UNSPECIFIED
	StringArrayValue *StringArray `json:"stringArrayValue,omitempty"`
	// StringValue: A variable-length string value. If this field is set,
	// value_type must be set to STRING or VALUE_TYPE_UNSPECIFIED
	StringValue string `json:"stringValue,omitempty"`
	// TemplateVariable: The placeholder text that can be referenced in a filter
	// string or MQL query. If omitted, the dashboard filter will be applied to all
	// relevant widgets in the dashboard.
	TemplateVariable string `json:"templateVariable,omitempty"`
	// TimeSeriesQuery: A query to run to fetch possible values for the filter.
	// Only OpsAnalyticsQueries are supported
	TimeSeriesQuery *TimeSeriesQuery `json:"timeSeriesQuery,omitempty"`
	// ValueType: The type of the filter value. If value_type is not provided, it
	// will be inferred from the default_value. If neither value_type nor
	// default_value is provided, value_type will be set to STRING by default.
	//
	// Possible values:
	//   "VALUE_TYPE_UNSPECIFIED" - Value type is unspecified
	//   "STRING" - String type
	//   "STRING_ARRAY" - String array type
	ValueType string `json:"valueType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FilterType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FilterType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DashboardFilter) MarshalJSON() ([]byte, error) {
	type NoMethod DashboardFilter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DataSet: Groups a time series query definition with charting options.
type DataSet struct {
	// Breakdowns: Optional. The collection of breakdowns to be applied to the
	// dataset.
	Breakdowns []*Breakdown `json:"breakdowns,omitempty"`
	// Dimensions: Optional. A collection of dimension columns.
	Dimensions []*Dimension `json:"dimensions,omitempty"`
	// LegendTemplate: A template string for naming TimeSeries in the resulting
	// data set. This should be a string with interpolations of the form
	// ${label_name}, which will resolve to the label's value.
	LegendTemplate string `json:"legendTemplate,omitempty"`
	// Measures: Optional. A collection of measures.
	Measures []*Measure `json:"measures,omitempty"`
	// MinAlignmentPeriod: Optional. The lower bound on data point frequency for
	// this data set, implemented by specifying the minimum alignment period to use
	// in a time series query For example, if the data is published once every 10
	// minutes, the min_alignment_period should be at least 10 minutes. It would
	// not make sense to fetch and align data at one minute intervals.
	MinAlignmentPeriod string `json:"minAlignmentPeriod,omitempty"`
	// PlotType: How this data should be plotted on the chart.
	//
	// Possible values:
	//   "PLOT_TYPE_UNSPECIFIED" - Plot type is unspecified. The view will default
	// to LINE.
	//   "LINE" - The data is plotted as a set of lines (one line per series).
	//   "STACKED_AREA" - The data is plotted as a set of filled areas (one area
	// per series), with the areas stacked vertically (the base of each area is the
	// top of its predecessor, and the base of the first area is the x-axis). Since
	// the areas do not overlap, each is filled with a different opaque color.
	//   "STACKED_BAR" - The data is plotted as a set of rectangular boxes (one box
	// per series), with the boxes stacked vertically (the base of each box is the
	// top of its predecessor, and the base of the first box is the x-axis). Since
	// the boxes do not overlap, each is filled with a different opaque color.
	//   "HEATMAP" - The data is plotted as a heatmap. The series being plotted
	// must have a DISTRIBUTION value type. The value of each bucket in the
	// distribution is displayed as a color. This type is not currently available
	// in the Stackdriver Monitoring application.
	PlotType string `json:"plotType,omitempty"`
	// Sort: Optional. A collection of sort options, affects the order of the data
	// and legend.
	Sort []*ColumnSortingOptions `json:"sort,omitempty"`
	// TargetAxis: Optional. The target axis to use for plotting the metric.
	//
	// Possible values:
	//   "TARGET_AXIS_UNSPECIFIED" - The target axis was not specified. Defaults to
	// Y1.
	//   "Y1" - The y_axis (the right axis of chart).
	//   "Y2" - The y2_axis (the left axis of chart).
	TargetAxis string `json:"targetAxis,omitempty"`
	// TimeSeriesQuery: Required. Fields for querying time series data from the
	// Stackdriver metrics API.
	TimeSeriesQuery *TimeSeriesQuery `json:"timeSeriesQuery,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Breakdowns") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Breakdowns") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataSet) MarshalJSON() ([]byte, error) {
	type NoMethod DataSet
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Dimension: A chart dimension. Dimensions are a structured label, class, or
// category for a set of measurements in your data.
type Dimension struct {
	// Column: Required. The name of the column in the source SQL query that is
	// used to chart the dimension.
	Column string `json:"column,omitempty"`
	// ColumnType: Optional. The type of the dimension column. This is relevant
	// only if one of the bin_size fields is set. If it is empty, the type
	// TIMESTAMP or INT64 will be assumed based on which bin_size field is set. If
	// populated, this should be set to one of the following types: DATE, TIME,
	// DATETIME, TIMESTAMP, BIGNUMERIC, INT64, NUMERIC, FLOAT64.
	ColumnType string `json:"columnType,omitempty"`
	// FloatBinSize: Optional. float_bin_size is used when the column type used for
	// a dimension is a floating point numeric column.
	FloatBinSize float64 `json:"floatBinSize,omitempty"`
	// MaxBinCount: A limit to the number of bins generated. When 0 is specified,
	// the maximum count is not enforced.
	MaxBinCount int64 `json:"maxBinCount,omitempty"`
	// NumericBinSize: numeric_bin_size is used when the column type used for a
	// dimension is numeric or string.
	NumericBinSize int64 `json:"numericBinSize,omitempty"`
	// SortColumn: The column name to sort on for binning. This column can be the
	// same column as this dimension or any other column used as a measure in the
	// results. If sort_order is set to NONE, then this value is not used.
	SortColumn string `json:"sortColumn,omitempty"`
	// SortOrder: The sort order applied to the sort column.
	//
	// Possible values:
	//   "SORT_ORDER_UNSPECIFIED" - An unspecified sort order. This option is
	// invalid when sorting is required.
	//   "SORT_ORDER_NONE" - No sorting is applied.
	//   "SORT_ORDER_ASCENDING" - The lowest-valued entries are selected first.
	//   "SORT_ORDER_DESCENDING" - The highest-valued entries are selected first.
	SortOrder string `json:"sortOrder,omitempty"`
	// TimeBinSize: time_bin_size is used when the data type specified by column is
	// a time type and the bin size is determined by a time duration. If
	// column_type is DATE, this must be a whole value multiple of 1 day. If
	// column_type is TIME, this must be less than or equal to 24 hours.
	TimeBinSize string `json:"timeBinSize,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Column") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Column") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Dimension) MarshalJSON() ([]byte, error) {
	type NoMethod Dimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Dimension) UnmarshalJSON(data []byte) error {
	type NoMethod Dimension
	var s1 struct {
		FloatBinSize gensupport.JSONFloat64 `json:"floatBinSize"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.FloatBinSize = float64(s1.FloatBinSize)
	return nil
}

// DroppedLabels: A set of (label, value) pairs that were removed from a
// Distribution time series during aggregation and then added as an attachment
// to a Distribution.Exemplar.The full label set for the exemplars is
// constructed by using the dropped pairs in combination with the label values
// that remain on the aggregated Distribution time series. The constructed full
// label set can be used to identify the specific entity, such as the instance
// or job, which might be contributing to a long-tail. However, with dropped
// labels, the storage requirements are reduced because only the aggregated
// distribution values for a large group of time series are stored.Note that
// there are no guarantees on ordering of the labels from exemplar-to-exemplar
// and from distribution-to-distribution in the same stream, and there may be
// duplicates. It is up to clients to resolve any ambiguities.
type DroppedLabels struct {
	// Label: Map from label to its value, for all labels dropped in any
	// aggregation.
	Label map[string]string `json:"label,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Label") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Label") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DroppedLabels) MarshalJSON() ([]byte, error) {
	type NoMethod DroppedLabels
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// ErrorReportingPanel: A widget that displays a list of error groups.
type ErrorReportingPanel struct {
	// ProjectNames: The resource name of the Google Cloud Platform project.
	// Written as projects/{projectID} or projects/{projectNumber}, where
	// {projectID} and {projectNumber} can be found in the Google Cloud console
	// (https://support.google.com/cloud/answer/6158840).Examples:
	// projects/my-project-123, projects/5551234.
	ProjectNames []string `json:"projectNames,omitempty"`
	// Services: An identifier of the service, such as the name of the executable,
	// job, or Google App Engine service name. This field is expected to have a low
	// number of values that are relatively stable over time, as opposed to
	// version, which can be changed whenever new code is deployed.Contains the
	// service name for error reports extracted from Google App Engine logs or
	// default if the App Engine default service is used.
	Services []string `json:"services,omitempty"`
	// Versions: Represents the source code version that the developer provided,
	// which could represent a version label or a Git SHA-1 hash, for example. For
	// App Engine standard environment, the version is set to the version of the
	// app.
	Versions []string `json:"versions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ProjectNames") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ProjectNames") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ErrorReportingPanel) MarshalJSON() ([]byte, error) {
	type NoMethod ErrorReportingPanel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EventAnnotation: Annotation configuration for one event type on a dashboard
type EventAnnotation struct {
	// DisplayName: Solely for UI display. Should not be used programmatically.
	DisplayName string `json:"displayName,omitempty"`
	// Enabled: Whether or not to show the events on the dashboard by default
	Enabled bool `json:"enabled,omitempty"`
	// EventType: The type of event to display.
	//
	// Possible values:
	//   "EVENT_TYPE_UNSPECIFIED" - No event type specified.
	//   "GKE_WORKLOAD_DEPLOYMENT" - Patch/update of GKE workload.
	//   "GKE_POD_CRASH" - Crash events for a GKE Pod.
	//   "GKE_POD_UNSCHEDULABLE" - Scheduling failures for GKE Pods.
	//   "GKE_CONTAINER_CREATION_FAILED" - Failure to create a GKE container.
	//   "GKE_CLUSTER_CREATE_DELETE" - Create/delete of a GKE cluster.
	//   "GKE_CLUSTER_UPDATE" - Update of a GKE cluster.
	//   "GKE_NODE_POOL_UPDATE" - Update of a GKE node pool.
	//   "GKE_CLUSTER_AUTOSCALER" - GKE cluster autoscaler event.
	//   "GKE_POD_AUTOSCALER" - GKE pod autoscaler event.
	//   "VM_TERMINATION" - Termination of a virtual machine.
	//   "VM_GUEST_OS_ERROR" - Guest OS error on a virtual machine.
	//   "VM_START_FAILED" - Start failure on a virtual machine.
	//   "MIG_UPDATE" - Update of a managed instance group.
	//   "MIG_AUTOSCALER" - Autoscaler event for a managed instance group.
	//   "CLOUD_RUN_DEPLOYMENT" - New deployment of a Cloud Run service.
	//   "CLOUD_SQL_FAILOVER" - Failover of a Cloud SQL instance.
	//   "CLOUD_SQL_START_STOP" - Start/stop of a Cloud SQL instance.
	//   "CLOUD_SQL_STORAGE" - Storage event for a Cloud SQL instance.
	//   "UPTIME_CHECK_FAILURE" - Failure of a Cloud Monitoring uptime check.
	//   "CLOUD_ALERTING_ALERT" - Alerts from Cloud Alerting
	//   "SERVICE_HEALTH_INCIDENT" - Incidents from Service Health
	//   "SAP_BACKINT" - Agent for SAP Backint related events.
	EventType string `json:"eventType,omitempty"`
	// Filter: string filtering the events - event dependant. Example values:
	// "resource.labels.pod_name = 'pod-1'"
	// "protoPayload.authenticationInfo.principalEmail='user@example.com'"
	Filter string `json:"filter,omitempty"`
	// ResourceNames: Per annotation level override for the names of logging
	// resources to search for events. Currently only projects are supported. If
	// both this field and the per annotation field is empty, it will default to
	// the host project. Limit: 50 projects. For example:
	// “projects/another-project-id”
	ResourceNames []string `json:"resourceNames,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EventAnnotation) MarshalJSON() ([]byte, error) {
	type NoMethod EventAnnotation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Field: A single field of a message type.
type Field struct {
	// Cardinality: The field cardinality.
	//
	// Possible values:
	//   "CARDINALITY_UNKNOWN" - For fields with unknown cardinality.
	//   "CARDINALITY_OPTIONAL" - For optional fields.
	//   "CARDINALITY_REQUIRED" - For required fields. Proto2 syntax only.
	//   "CARDINALITY_REPEATED" - For repeated fields.
	Cardinality string `json:"cardinality,omitempty"`
	// DefaultValue: The string value of the default value of this field. Proto2
	// syntax only.
	DefaultValue string `json:"defaultValue,omitempty"`
	// JsonName: The field JSON name.
	JsonName string `json:"jsonName,omitempty"`
	// Kind: The field type.
	//
	// Possible values:
	//   "TYPE_UNKNOWN" - Field type unknown.
	//   "TYPE_DOUBLE" - Field type double.
	//   "TYPE_FLOAT" - Field type float.
	//   "TYPE_INT64" - Field type int64.
	//   "TYPE_UINT64" - Field type uint64.
	//   "TYPE_INT32" - Field type int32.
	//   "TYPE_FIXED64" - Field type fixed64.
	//   "TYPE_FIXED32" - Field type fixed32.
	//   "TYPE_BOOL" - Field type bool.
	//   "TYPE_STRING" - Field type string.
	//   "TYPE_GROUP" - Field type group. Proto2 syntax only, and deprecated.
	//   "TYPE_MESSAGE" - Field type message.
	//   "TYPE_BYTES" - Field type bytes.
	//   "TYPE_UINT32" - Field type uint32.
	//   "TYPE_ENUM" - Field type enum.
	//   "TYPE_SFIXED32" - Field type sfixed32.
	//   "TYPE_SFIXED64" - Field type sfixed64.
	//   "TYPE_SINT32" - Field type sint32.
	//   "TYPE_SINT64" - Field type sint64.
	Kind string `json:"kind,omitempty"`
	// Name: The field name.
	Name string `json:"name,omitempty"`
	// Number: The field number.
	Number int64 `json:"number,omitempty"`
	// OneofIndex: The index of the field type in Type.oneofs, for message or
	// enumeration types. The first type has index 1; zero means the type is not in
	// the list.
	OneofIndex int64 `json:"oneofIndex,omitempty"`
	// Options: The protocol buffer options.
	Options []*Option `json:"options,omitempty"`
	// Packed: Whether to use alternative packed wire representation.
	Packed bool `json:"packed,omitempty"`
	// TypeUrl: The field type URL, without the scheme, for message or enumeration
	// types. Example: "type.googleapis.com/google.protobuf.Timestamp".
	TypeUrl string `json:"typeUrl,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Cardinality") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Cardinality") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Field) MarshalJSON() ([]byte, error) {
	type NoMethod Field
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GaugeView: A gauge chart shows where the current value sits within a
// pre-defined range. The upper and lower bounds should define the possible
// range of values for the scorecard's query (inclusive).
type GaugeView struct {
	// LowerBound: The lower bound for this gauge chart. The value of the chart
	// should always be greater than or equal to this.
	LowerBound float64 `json:"lowerBound,omitempty"`
	// UpperBound: The upper bound for this gauge chart. The value of the chart
	// should always be less than or equal to this.
	UpperBound float64 `json:"upperBound,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LowerBound") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LowerBound") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GaugeView) MarshalJSON() ([]byte, error) {
	type NoMethod GaugeView
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *GaugeView) UnmarshalJSON(data []byte) error {
	type NoMethod GaugeView
	var s1 struct {
		LowerBound gensupport.JSONFloat64 `json:"lowerBound"`
		UpperBound gensupport.JSONFloat64 `json:"upperBound"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.LowerBound = float64(s1.LowerBound)
	s.UpperBound = float64(s1.UpperBound)
	return nil
}

// GridLayout: A basic layout divides the available space into vertical columns
// of equal width and arranges a list of widgets using a row-first strategy.
type GridLayout struct {
	// Columns: The number of columns into which the view's width is divided. If
	// omitted or set to zero, a system default will be used while rendering.
	Columns int64 `json:"columns,omitempty,string"`
	// Widgets: The informational elements that are arranged into the columns
	// row-first.
	Widgets []*Widget `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Columns") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Columns") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GridLayout) MarshalJSON() ([]byte, error) {
	type NoMethod GridLayout
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HttpBody: Message that represents an arbitrary HTTP body. It should only be
// used for payload formats that can't be represented as JSON, such as raw
// binary or an HTML page.This message can be used both in streaming and
// non-streaming API methods in the request as well as the response.It can be
// used as a top-level request field, which is convenient if one wants to
// extract parameters from either the URL or HTTP template into the request
// fields and also want access to the raw HTTP body.Example: message
// GetResourceRequest { // A unique request id. string request_id = 1; // The
// raw HTTP body is bound to this field. google.api.HttpBody http_body = 2; }
// service ResourceService { rpc GetResource(GetResourceRequest) returns
// (google.api.HttpBody); rpc UpdateResource(google.api.HttpBody) returns
// (google.protobuf.Empty); } Example with streaming methods: service
// CaldavService { rpc GetCalendar(stream google.api.HttpBody) returns (stream
// google.api.HttpBody); rpc UpdateCalendar(stream google.api.HttpBody) returns
// (stream google.api.HttpBody); } Use of this type only changes how the
// request and response bodies are handled, all other features will continue to
// work unchanged.
type HttpBody struct {
	// ContentType: The HTTP Content-Type header value specifying the content type
	// of the body.
	ContentType string `json:"contentType,omitempty"`
	// Data: The HTTP request/response body as raw binary.
	Data any `json:"data,omitempty"`
	// Extensions: Application specific response metadata. Must be set in the first
	// response for streaming APIs.
	Extensions []googleapi.RawMessage `json:"extensions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HttpBody) MarshalJSON() ([]byte, error) {
	type NoMethod HttpBody
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IncidentList: A widget that displays a list of incidents
type IncidentList struct {
	// MonitoredResources: Optional. The monitored resource for which incidents are
	// listed. The resource doesn't need to be fully specified. That is, you can
	// specify the resource type but not the values of the resource labels. The
	// resource type and labels are used for filtering.
	MonitoredResources []*MonitoredResource `json:"monitoredResources,omitempty"`
	// PolicyNames: Optional. A list of alert policy names to filter the incident
	// list by. Don't include the project ID prefix in the policy name. For
	// example, use alertPolicies/utilization.
	PolicyNames []string `json:"policyNames,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MonitoredResources") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MonitoredResources") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IncidentList) MarshalJSON() ([]byte, error) {
	type NoMethod IncidentList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Interval: Represents a time interval, encoded as a Timestamp start
// (inclusive) and a Timestamp end (exclusive).The start must be less than or
// equal to the end. When the start equals the end, the interval is empty
// (matches no time). When both start and end are unspecified, the interval
// matches any time.
type Interval struct {
	// EndTime: Optional. Exclusive end of the interval.If specified, a Timestamp
	// matching this interval will have to be before the end.
	EndTime string `json:"endTime,omitempty"`
	// StartTime: Optional. Inclusive start of the interval.If specified, a
	// Timestamp matching this interval will have to be the same or after the
	// start.
	StartTime string `json:"startTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Interval) MarshalJSON() ([]byte, error) {
	type NoMethod Interval
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListDashboardsResponse: The ListDashboards request.
type ListDashboardsResponse struct {
	// Dashboards: The list of requested dashboards.
	Dashboards []*Dashboard `json:"dashboards,omitempty"`
	// NextPageToken: If there are more results than have been returned, then this
	// field is set to a non-empty value. To see the additional results, use that
	// value as page_token in the next call to this method.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Dashboards") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Dashboards") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListDashboardsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListDashboardsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListMetricsScopesByMonitoredProjectResponse: Response for the
// ListMetricsScopesByMonitoredProject method.
type ListMetricsScopesByMonitoredProjectResponse struct {
	// MetricsScopes: A set of all metrics scopes that the specified monitored
	// project has been added to.
	MetricsScopes []*MetricsScope `json:"metricsScopes,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "MetricsScopes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MetricsScopes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListMetricsScopesByMonitoredProjectResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListMetricsScopesByMonitoredProjectResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LogsPanel: A widget that displays a stream of log.
type LogsPanel struct {
	// Filter: A filter that chooses which log entries to return. See Advanced Logs
	// Queries (https://cloud.google.com/logging/docs/view/advanced-queries). Only
	// log entries that match the filter are returned. An empty filter matches all
	// log entries.
	Filter string `json:"filter,omitempty"`
	// ResourceNames: The names of logging resources to collect logs for. Currently
	// projects and storage views are supported. If empty, the widget will default
	// to the host project.
	ResourceNames []string `json:"resourceNames,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Filter") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Filter") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LogsPanel) MarshalJSON() ([]byte, error) {
	type NoMethod LogsPanel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Measure: A chart measure. Measures represent a measured property in your
// chart data such as rainfall in inches, number of units sold, revenue gained,
// etc.
type Measure struct {
	// AggregationFunction: Required. The aggregation function applied to the input
	// column. This must not be set to "none" unless binning is disabled on the
	// dimension. The aggregation function is used to group points on the dimension
	// bins.
	AggregationFunction *AggregationFunction `json:"aggregationFunction,omitempty"`
	// Column: Required. The column name within in the dataset used for the
	// measure.
	Column string `json:"column,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AggregationFunction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AggregationFunction") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Measure) MarshalJSON() ([]byte, error) {
	type NoMethod Measure
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MetricsScope: Represents a Metrics Scope
// (https://cloud.google.com/monitoring/settings#concept-scope) in Cloud
// Monitoring, which specifies one or more Google projects and zero or more AWS
// accounts to monitor together.
type MetricsScope struct {
	// CreateTime: Output only. The time when this Metrics Scope was created.
	CreateTime string `json:"createTime,omitempty"`
	// MonitoredProjects: Output only. The list of projects monitored by this
	// Metrics Scope.
	MonitoredProjects []*MonitoredProject `json:"monitoredProjects,omitempty"`
	// Name: Immutable. The resource name of the Monitoring Metrics Scope. On
	// input, the resource name can be specified with the scoping project ID or
	// number. On output, the resource name is specified with the scoping project
	// number. Example:
	// locations/global/metricsScopes/{SCOPING_PROJECT_ID_OR_NUMBER}
	Name string `json:"name,omitempty"`
	// UpdateTime: Output only. The time when this Metrics Scope record was last
	// updated.
	UpdateTime string `json:"updateTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MetricsScope) MarshalJSON() ([]byte, error) {
	type NoMethod MetricsScope
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MonitoredProject: A project being monitored
// (https://cloud.google.com/monitoring/settings/multiple-projects#create-multi)
// by a Metrics Scope.
type MonitoredProject struct {
	// CreateTime: Output only. The time when this MonitoredProject was created.
	CreateTime string `json:"createTime,omitempty"`
	// Name: Immutable. The resource name of the MonitoredProject. On input, the
	// resource name includes the scoping project ID and monitored project ID. On
	// output, it contains the equivalent project numbers. Example:
	// locations/global/metricsScopes/{SCOPING_PROJECT_ID_OR_NUMBER}/projects/{MONIT
	// ORED_PROJECT_ID_OR_NUMBER}
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MonitoredProject) MarshalJSON() ([]byte, error) {
	type NoMethod MonitoredProject
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MonitoredResource: An object representing a resource that can be used for
// monitoring, logging, billing, or other purposes. Examples include virtual
// machine instances, databases, and storage devices such as disks. The type
// field identifies a MonitoredResourceDescriptor object that describes the
// resource's schema. Information in the labels field identifies the actual
// resource and its attributes according to the schema. For example, a
// particular Compute Engine VM instance could be represented by the following
// object, because the MonitoredResourceDescriptor for "gce_instance" has
// labels "project_id", "instance_id" and "zone": { "type": "gce_instance",
// "labels": { "project_id": "my-project", "instance_id": "12345678901234",
// "zone": "us-central1-a" }}
type MonitoredResource struct {
	// Labels: Required. Values for all of the labels listed in the associated
	// monitored resource descriptor. For example, Compute Engine VM instances use
	// the labels "project_id", "instance_id", and "zone".
	Labels map[string]string `json:"labels,omitempty"`
	// Type: Required. The monitored resource type. This field must match the type
	// field of a MonitoredResourceDescriptor object. For example, the type of a
	// Compute Engine VM instance is gce_instance. For a list of types, see
	// Monitoring resource types
	// (https://cloud.google.com/monitoring/api/resources) and Logging resource
	// types (https://cloud.google.com/logging/docs/api/v2/resource-list).
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Labels") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Labels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MonitoredResource) MarshalJSON() ([]byte, error) {
	type NoMethod MonitoredResource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MosaicLayout: A mosaic layout divides the available space into a grid of
// blocks, and overlays the grid with tiles. Unlike GridLayout, tiles may span
// multiple grid blocks and can be placed at arbitrary locations in the grid.
type MosaicLayout struct {
	// Columns: The number of columns in the mosaic grid. The number of columns
	// must be between 1 and 48, inclusive.
	Columns int64 `json:"columns,omitempty"`
	// Tiles: The tiles to display.
	Tiles []*Tile `json:"tiles,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Columns") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Columns") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MosaicLayout) MarshalJSON() ([]byte, error) {
	type NoMethod MosaicLayout
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Operation: This resource represents a long-running operation that is the
// result of a network API call.
type Operation struct {
	// Done: If the value is false, it means the operation is still in progress. If
	// true, the operation is completed, and either error or response is available.
	Done bool `json:"done,omitempty"`
	// Error: The error result of the operation in case of failure or cancellation.
	Error *Status `json:"error,omitempty"`
	// Metadata: Service-specific metadata associated with the operation. It
	// typically contains progress information and common metadata such as create
	// time. Some services might not provide such metadata. Any method that returns
	// a long-running operation should document the metadata type, if any.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: The server-assigned name, which is only unique within the same service
	// that originally returns it. If you use the default HTTP mapping, the name
	// should be a resource name ending with operations/{unique_id}.
	Name string `json:"name,omitempty"`
	// Response: The normal, successful response of the operation. If the original
	// method returns no data on success, such as Delete, the response is
	// google.protobuf.Empty. If the original method is standard Get/Create/Update,
	// the response should be the resource. For other methods, the response should
	// have the type XxxResponse, where Xxx is the original method name. For
	// example, if the original method name is TakeSnapshot(), the inferred
	// response type is TakeSnapshotResponse.
	Response googleapi.RawMessage `json:"response,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Done") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Done") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Operation) MarshalJSON() ([]byte, error) {
	type NoMethod Operation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OperationMetadata: Contains metadata for longrunning operation for the edit
// Metrics Scope endpoints.
type OperationMetadata struct {
	// CreateTime: The time when the batch request was received.
	CreateTime string `json:"createTime,omitempty"`
	// State: Current state of the batch operation.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Invalid.
	//   "CREATED" - Request has been received.
	//   "RUNNING" - Request is actively being processed.
	//   "DONE" - The batch processing is done.
	//   "CANCELLED" - The batch processing was cancelled.
	State string `json:"state,omitempty"`
	// UpdateTime: The time when the operation result was last updated.
	UpdateTime string `json:"updateTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OperationMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod OperationMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OpsAnalyticsQuery: Preview: A query that produces an aggregated response and
// supporting data. This is a preview feature and may be subject to change
// before final release.
type OpsAnalyticsQuery struct {
	// Sql: A SQL query to fetch time series, category series, or numeric series
	// data.
	Sql string `json:"sql,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Sql") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Sql") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OpsAnalyticsQuery) MarshalJSON() ([]byte, error) {
	type NoMethod OpsAnalyticsQuery
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Option: A protocol buffer option, which can be attached to a message, field,
// enumeration, etc.
type Option struct {
	// Name: The option's name. For protobuf built-in options (options defined in
	// descriptor.proto), this is the short name. For example, "map_entry". For
	// custom options, it should be the fully-qualified name. For example,
	// "google.api.http".
	Name string `json:"name,omitempty"`
	// Value: The option's value packed in an Any message. If the value is a
	// primitive, the corresponding wrapper type defined in
	// google/protobuf/wrappers.proto should be used. If the value is an enum, it
	// should be stored as an int32 value using the google.protobuf.Int32Value
	// type.
	Value googleapi.RawMessage `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Option) MarshalJSON() ([]byte, error) {
	type NoMethod Option
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Parameter: Preview: Parameter value applied to the aggregation function.
// This is a preview feature and may be subject to change before final release.
type Parameter struct {
	// DoubleValue: A floating-point parameter value.
	DoubleValue float64 `json:"doubleValue,omitempty"`
	// IntValue: An integer parameter value.
	IntValue int64 `json:"intValue,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "DoubleValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DoubleValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Parameter) MarshalJSON() ([]byte, error) {
	type NoMethod Parameter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Parameter) UnmarshalJSON(data []byte) error {
	type NoMethod Parameter
	var s1 struct {
		DoubleValue gensupport.JSONFloat64 `json:"doubleValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.DoubleValue = float64(s1.DoubleValue)
	return nil
}

// PickTimeSeriesFilter: Describes a ranking-based time series filter. Each
// input time series is ranked with an aligner. The filter will allow up to
// num_time_series time series to pass through it, selecting them based on the
// relative ranking.For example, if ranking_method is METHOD_MEAN,direction is
// BOTTOM, and num_time_series is 3, then the 3 times series with the lowest
// mean values will pass through the filter.
type PickTimeSeriesFilter struct {
	// Direction: How to use the ranking to select time series that pass through
	// the filter.
	//
	// Possible values:
	//   "DIRECTION_UNSPECIFIED" - Not allowed. You must specify a different
	// Direction if you specify a PickTimeSeriesFilter.
	//   "TOP" - Pass the highest num_time_series ranking inputs.
	//   "BOTTOM" - Pass the lowest num_time_series ranking inputs.
	Direction string `json:"direction,omitempty"`
	// Interval: Select the top N streams/time series within this time interval
	Interval *Interval `json:"interval,omitempty"`
	// NumTimeSeries: How many time series to allow to pass through the filter.
	NumTimeSeries int64 `json:"numTimeSeries,omitempty"`
	// RankingMethod: ranking_method is applied to each time series independently
	// to produce the value which will be used to compare the time series to other
	// time series.
	//
	// Possible values:
	//   "METHOD_UNSPECIFIED" - Not allowed. You must specify a different Method if
	// you specify a PickTimeSeriesFilter.
	//   "METHOD_MEAN" - Select the mean of all values.
	//   "METHOD_MAX" - Select the maximum value.
	//   "METHOD_MIN" - Select the minimum value.
	//   "METHOD_SUM" - Compute the sum of all values.
	//   "METHOD_LATEST" - Select the most recent value.
	RankingMethod string `json:"rankingMethod,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Direction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Direction") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PickTimeSeriesFilter) MarshalJSON() ([]byte, error) {
	type NoMethod PickTimeSeriesFilter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PieChart: A widget that displays timeseries data as a pie or a donut.
type PieChart struct {
	// ChartType: Required. Indicates the visualization type for the PieChart.
	//
	// Possible values:
	//   "PIE_CHART_TYPE_UNSPECIFIED" - The zero value. No type specified. Do not
	// use.
	//   "PIE" - A Pie type PieChart.
	//   "DONUT" - Similar to PIE, but the DONUT type PieChart has a hole in the
	// middle.
	ChartType string `json:"chartType,omitempty"`
	// DataSets: Required. The queries for the chart's data.
	DataSets []*PieChartDataSet `json:"dataSets,omitempty"`
	// ShowLabels: Optional. Indicates whether or not the pie chart should show
	// slices' labels
	ShowLabels bool `json:"showLabels,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChartType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChartType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PieChart) MarshalJSON() ([]byte, error) {
	type NoMethod PieChart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PieChartDataSet: Groups a time series query definition.
type PieChartDataSet struct {
	// Dimensions: A dimension is a structured label, class, or category for a set
	// of measurements in your data.
	Dimensions []*Dimension `json:"dimensions,omitempty"`
	// Measures: A measure is a measured value of a property in your data. For
	// example, rainfall in inches, number of units sold, revenue gained, etc.
	Measures []*Measure `json:"measures,omitempty"`
	// MinAlignmentPeriod: Optional. The lower bound on data point frequency for
	// this data set, implemented by specifying the minimum alignment period to use
	// in a time series query. For example, if the data is published once every 10
	// minutes, the min_alignment_period should be at least 10 minutes. It would
	// not make sense to fetch and align data at one minute intervals.
	MinAlignmentPeriod string `json:"minAlignmentPeriod,omitempty"`
	// SliceNameTemplate: Optional. A template for the name of the slice. This name
	// will be displayed in the legend and the tooltip of the pie chart. It
	// replaces the auto-generated names for the slices. For example, if the
	// template is set to ${resource.labels.zone}, the zone's value will be used
	// for the name instead of the default name.
	SliceNameTemplate string `json:"sliceNameTemplate,omitempty"`
	// TimeSeriesQuery: Required. The query for the PieChart. See,
	// google.monitoring.dashboard.v1.TimeSeriesQuery.
	TimeSeriesQuery *TimeSeriesQuery `json:"timeSeriesQuery,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Dimensions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Dimensions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PieChartDataSet) MarshalJSON() ([]byte, error) {
	type NoMethod PieChartDataSet
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryExemplarsRequest: QueryExemplarsRequest holds all parameters of the
// Prometheus upstream API for querying exemplars.
type QueryExemplarsRequest struct {
	// End: The end time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	End string `json:"end,omitempty"`
	// Query: A PromQL query string. Query language documentation:
	// https://prometheus.io/docs/prometheus/latest/querying/basics/.
	Query string `json:"query,omitempty"`
	// Start: The start time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	Start string `json:"start,omitempty"`
	// ForceSendFields is a list of field names (e.g. "End") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "End") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryExemplarsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QueryExemplarsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryInstantRequest: QueryInstantRequest holds all parameters of the
// Prometheus upstream instant query API plus GCM specific parameters.
type QueryInstantRequest struct {
	// Query: A PromQL query string. Query language documentation:
	// https://prometheus.io/docs/prometheus/latest/querying/basics/.
	Query string `json:"query,omitempty"`
	// Time: The single point in time to evaluate the query for. Either floating
	// point UNIX seconds or RFC3339 formatted timestamp.
	Time string `json:"time,omitempty"`
	// Timeout: An upper bound timeout for the query. Either a Prometheus duration
	// string
	// (https://prometheus.io/docs/prometheus/latest/querying/basics/#time-durations)
	// or floating point seconds. This non-standard encoding must be used for
	// compatibility with the open source API. Clients may still implement timeouts
	// at the connection level while ignoring this field.
	Timeout string `json:"timeout,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Query") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Query") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryInstantRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QueryInstantRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryLabelsRequest: QueryLabelsRequest holds all parameters of the
// Prometheus upstream API for returning a list of label names.
type QueryLabelsRequest struct {
	// End: The end time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	End string `json:"end,omitempty"`
	// Match: A list of matchers encoded in the Prometheus label matcher format to
	// constrain the values to series that satisfy them.
	Match string `json:"match,omitempty"`
	// Start: The start time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	Start string `json:"start,omitempty"`
	// ForceSendFields is a list of field names (e.g. "End") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "End") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryLabelsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QueryLabelsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryRangeRequest: QueryRangeRequest holds all parameters of the Prometheus
// upstream range query API plus GCM specific parameters.
type QueryRangeRequest struct {
	// End: The end time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	End string `json:"end,omitempty"`
	// Query: A PromQL query string. Query language documentation:
	// https://prometheus.io/docs/prometheus/latest/querying/basics/.
	Query string `json:"query,omitempty"`
	// Start: The start time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	Start string `json:"start,omitempty"`
	// Step: The resolution of query result. Either a Prometheus duration string
	// (https://prometheus.io/docs/prometheus/latest/querying/basics/#time-durations)
	// or floating point seconds. This non-standard encoding must be used for
	// compatibility with the open source API. Clients may still implement timeouts
	// at the connection level while ignoring this field.
	Step string `json:"step,omitempty"`
	// Timeout: An upper bound timeout for the query. Either a Prometheus duration
	// string
	// (https://prometheus.io/docs/prometheus/latest/querying/basics/#time-durations)
	// or floating point seconds. This non-standard encoding must be used for
	// compatibility with the open source API. Clients may still implement timeouts
	// at the connection level while ignoring this field.
	Timeout string `json:"timeout,omitempty"`
	// ForceSendFields is a list of field names (e.g. "End") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "End") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryRangeRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QueryRangeRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QuerySeriesRequest: QuerySeries holds all parameters of the Prometheus
// upstream API for querying series.
type QuerySeriesRequest struct {
	// End: The end time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	End string `json:"end,omitempty"`
	// Start: The start time to evaluate the query for. Either floating point UNIX
	// seconds or RFC3339 formatted timestamp.
	Start string `json:"start,omitempty"`
	// ForceSendFields is a list of field names (e.g. "End") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "End") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QuerySeriesRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QuerySeriesRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RatioPart: Describes a query to build the numerator or denominator of a
// TimeSeriesFilterRatio.
type RatioPart struct {
	// Aggregation: By default, the raw time series data is returned. Use this
	// field to combine multiple time series for different views of the data.
	Aggregation *Aggregation `json:"aggregation,omitempty"`
	// Filter: Required. The monitoring filter
	// (https://cloud.google.com/monitoring/api/v3/filters) that identifies the
	// metric types, resources, and projects to query.
	Filter string `json:"filter,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aggregation") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aggregation") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RatioPart) MarshalJSON() ([]byte, error) {
	type NoMethod RatioPart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Row: Defines the layout properties and content for a row.
type Row struct {
	// Weight: The relative weight of this row. The row weight is used to adjust
	// the height of rows on the screen (relative to peers). Greater the weight,
	// greater the height of the row on the screen. If omitted, a value of 1 is
	// used while rendering.
	Weight int64 `json:"weight,omitempty,string"`
	// Widgets: The display widgets arranged horizontally in this row.
	Widgets []*Widget `json:"widgets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Weight") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Weight") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Row) MarshalJSON() ([]byte, error) {
	type NoMethod Row
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RowLayout: A simplified layout that divides the available space into rows
// and arranges a set of widgets horizontally in each row.
type RowLayout struct {
	// Rows: The rows of content to display.
	Rows []*Row `json:"rows,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Rows") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Rows") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RowLayout) MarshalJSON() ([]byte, error) {
	type NoMethod RowLayout
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Scorecard: A widget showing the latest value of a metric, and how this value
// relates to one or more thresholds.
type Scorecard struct {
	// BlankView: Will cause the Scorecard to show only the value, with no
	// indicator to its value relative to its thresholds.
	BlankView *Empty `json:"blankView,omitempty"`
	// Breakdowns: Optional. The collection of breakdowns to be applied to the
	// dataset. A breakdown is a way to slice the data. For example, you can break
	// down the data by region.
	Breakdowns []*Breakdown `json:"breakdowns,omitempty"`
	// Dimensions: Optional. A dimension is a structured label, class, or category
	// for a set of measurements in your data.
	Dimensions []*Dimension `json:"dimensions,omitempty"`
	// GaugeView: Will cause the scorecard to show a gauge chart.
	GaugeView *GaugeView `json:"gaugeView,omitempty"`
	// Measures: Optional. A measure is a measured value of a property in your
	// data. For example, rainfall in inches, number of units sold, revenue gained,
	// etc.
	Measures []*Measure `json:"measures,omitempty"`
	// SparkChartView: Will cause the scorecard to show a spark chart.
	SparkChartView *SparkChartView `json:"sparkChartView,omitempty"`
	// Thresholds: The thresholds used to determine the state of the scorecard
	// given the time series' current value. For an actual value x, the scorecard
	// is in a danger state if x is less than or equal to a danger threshold that
	// triggers below, or greater than or equal to a danger threshold that triggers
	// above. Similarly, if x is above/below a warning threshold that triggers
	// above/below, then the scorecard is in a warning state - unless x also puts
	// it in a danger state. (Danger trumps warning.)As an example, consider a
	// scorecard with the following four thresholds: { value: 90, category:
	// 'DANGER', trigger: 'ABOVE', }, { value: 70, category: 'WARNING', trigger:
	// 'ABOVE', }, { value: 10, category: 'DANGER', trigger: 'BELOW', }, { value:
	// 20, category: 'WARNING', trigger: 'BELOW', } Then: values less than or equal
	// to 10 would put the scorecard in a DANGER state, values greater than 10 but
	// less than or equal to 20 a WARNING state, values strictly between 20 and 70
	// an OK state, values greater than or equal to 70 but less than 90 a WARNING
	// state, and values greater than or equal to 90 a DANGER state.
	Thresholds []*Threshold `json:"thresholds,omitempty"`
	// TimeSeriesQuery: Required. Fields for querying time series data from the
	// Stackdriver metrics API.
	TimeSeriesQuery *TimeSeriesQuery `json:"timeSeriesQuery,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BlankView") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BlankView") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Scorecard) MarshalJSON() ([]byte, error) {
	type NoMethod Scorecard
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SectionHeader: A widget that defines a new section header. Sections populate
// a table of contents and allow easier navigation of long-form content.
type SectionHeader struct {
	// DividerBelow: Whether to insert a divider below the section in the table of
	// contents
	DividerBelow bool `json:"dividerBelow,omitempty"`
	// Subtitle: The subtitle of the section
	Subtitle string `json:"subtitle,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DividerBelow") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DividerBelow") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SectionHeader) MarshalJSON() ([]byte, error) {
	type NoMethod SectionHeader
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SingleViewGroup: A widget that groups the other widgets by using a dropdown
// menu. All widgets that are within the area spanned by the grouping widget
// are considered member widgets.
type SingleViewGroup struct {
}

// SourceContext: SourceContext represents information about the source of a
// protobuf element, like the file in which it is defined.
type SourceContext struct {
	// FileName: The path-qualified name of the .proto file that contained the
	// associated protobuf element. For example:
	// "google/protobuf/source_context.proto".
	FileName string `json:"fileName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FileName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FileName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SourceContext) MarshalJSON() ([]byte, error) {
	type NoMethod SourceContext
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpanContext: The context of a span. This is attached to an Exemplar in
// Distribution values during aggregation.It contains the name of a span with
// format: projects/[PROJECT_ID_OR_NUMBER]/traces/[TRACE_ID]/spans/[SPAN_ID]
type SpanContext struct {
	// SpanName: The resource name of the span. The format is:
	// projects/[PROJECT_ID_OR_NUMBER]/traces/[TRACE_ID]/spans/[SPAN_ID] [TRACE_ID]
	// is a unique identifier for a trace within a project; it is a 32-character
	// hexadecimal encoding of a 16-byte array.[SPAN_ID] is a unique identifier for
	// a span within a trace; it is a 16-character hexadecimal encoding of an
	// 8-byte array.
	SpanName string `json:"spanName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "SpanName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SpanName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpanContext) MarshalJSON() ([]byte, error) {
	type NoMethod SpanContext
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SparkChartView: A sparkChart is a small chart suitable for inclusion in a
// table-cell or inline in text. This message contains the configuration for a
// sparkChart to show up on a Scorecard, showing recent trends of the
// scorecard's timeseries.
type SparkChartView struct {
	// MinAlignmentPeriod: The lower bound on data point frequency in the chart
	// implemented by specifying the minimum alignment period to use in a time
	// series query. For example, if the data is published once every 10 minutes it
	// would not make sense to fetch and align data at one minute intervals. This
	// field is optional and exists only as a hint.
	MinAlignmentPeriod string `json:"minAlignmentPeriod,omitempty"`
	// SparkChartType: Required. The type of sparkchart to show in this chartView.
	//
	// Possible values:
	//   "SPARK_CHART_TYPE_UNSPECIFIED" - Not allowed in well-formed requests.
	//   "SPARK_LINE" - The sparkline will be rendered as a small line chart.
	//   "SPARK_BAR" - The sparkbar will be rendered as a small bar chart.
	SparkChartType string `json:"sparkChartType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MinAlignmentPeriod") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MinAlignmentPeriod") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SparkChartView) MarshalJSON() ([]byte, error) {
	type NoMethod SparkChartView
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StatisticalTimeSeriesFilter: A filter that ranks streams based on their
// statistical relation to other streams in a request. Note: This field is
// deprecated and completely ignored by the API.
type StatisticalTimeSeriesFilter struct {
	// NumTimeSeries: How many time series to output.
	NumTimeSeries int64 `json:"numTimeSeries,omitempty"`
	// RankingMethod: rankingMethod is applied to a set of time series, and then
	// the produced value for each individual time series is used to compare a
	// given time series to others. These are methods that cannot be applied
	// stream-by-stream, but rather require the full context of a request to
	// evaluate time series.
	//
	// Possible values:
	//   "METHOD_UNSPECIFIED" - Not allowed in well-formed requests.
	//   "METHOD_CLUSTER_OUTLIER" - Compute the outlier score of each stream.
	RankingMethod string `json:"rankingMethod,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NumTimeSeries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NumTimeSeries") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StatisticalTimeSeriesFilter) MarshalJSON() ([]byte, error) {
	type NoMethod StatisticalTimeSeriesFilter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Status: The Status type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each Status message contains three
// pieces of data: error code, error message, and error details.You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StringArray: An array of strings
type StringArray struct {
	// Values: The values of the array
	Values []string `json:"values,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Values") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Values") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StringArray) MarshalJSON() ([]byte, error) {
	type NoMethod StringArray
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TableDataSet: Groups a time series query definition with table options.
type TableDataSet struct {
	// MinAlignmentPeriod: Optional. The lower bound on data point frequency for
	// this data set, implemented by specifying the minimum alignment period to use
	// in a time series query For example, if the data is published once every 10
	// minutes, the min_alignment_period should be at least 10 minutes. It would
	// not make sense to fetch and align data at one minute intervals.
	MinAlignmentPeriod string `json:"minAlignmentPeriod,omitempty"`
	// TableDisplayOptions: Optional. Table display options for configuring how the
	// table is rendered.
	TableDisplayOptions *TableDisplayOptions `json:"tableDisplayOptions,omitempty"`
	// TableTemplate: Optional. A template string for naming TimeSeries in the
	// resulting data set. This should be a string with interpolations of the form
	// ${label_name}, which will resolve to the label's value i.e.
	// "${resource.labels.project_id}."
	TableTemplate string `json:"tableTemplate,omitempty"`
	// TimeSeriesQuery: Required. Fields for querying time series data from the
	// Stackdriver metrics API.
	TimeSeriesQuery *TimeSeriesQuery `json:"timeSeriesQuery,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MinAlignmentPeriod") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MinAlignmentPeriod") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TableDataSet) MarshalJSON() ([]byte, error) {
	type NoMethod TableDataSet
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TableDisplayOptions: Table display options that can be reused.
type TableDisplayOptions struct {
	// ShownColumns: Optional. This field is unused and has been replaced by
	// TimeSeriesTable.column_settings
	ShownColumns []string `json:"shownColumns,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ShownColumns") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ShownColumns") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TableDisplayOptions) MarshalJSON() ([]byte, error) {
	type NoMethod TableDisplayOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Text: A widget that displays textual content.
type Text struct {
	// Content: The text content to be displayed.
	Content string `json:"content,omitempty"`
	// Format: How the text content is formatted.
	//
	// Possible values:
	//   "FORMAT_UNSPECIFIED" - Format is unspecified. Defaults to MARKDOWN.
	//   "MARKDOWN" - The text contains Markdown formatting.
	//   "RAW" - The text contains no special formatting.
	Format string `json:"format,omitempty"`
	// Style: How the text is styled
	Style *TextStyle `json:"style,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Text) MarshalJSON() ([]byte, error) {
	type NoMethod Text
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TextStyle: Properties that determine how the title and content are styled
type TextStyle struct {
	// BackgroundColor: The background color as a hex string. "#RRGGBB" or "#RGB"
	BackgroundColor string `json:"backgroundColor,omitempty"`
	// FontSize: Font sizes for both the title and content. The title will still be
	// larger relative to the content.
	//
	// Possible values:
	//   "FONT_SIZE_UNSPECIFIED" - No font size specified, will default to FS_LARGE
	//   "FS_EXTRA_SMALL" - Extra small font size
	//   "FS_SMALL" - Small font size
	//   "FS_MEDIUM" - Medium font size
	//   "FS_LARGE" - Large font size
	//   "FS_EXTRA_LARGE" - Extra large font size
	FontSize string `json:"fontSize,omitempty"`
	// HorizontalAlignment: The horizontal alignment of both the title and content
	//
	// Possible values:
	//   "HORIZONTAL_ALIGNMENT_UNSPECIFIED" - No horizontal alignment specified,
	// will default to H_LEFT
	//   "H_LEFT" - Left-align
	//   "H_CENTER" - Center-align
	//   "H_RIGHT" - Right-align
	HorizontalAlignment string `json:"horizontalAlignment,omitempty"`
	// Padding: The amount of padding around the widget
	//
	// Possible values:
	//   "PADDING_SIZE_UNSPECIFIED" - No padding size specified, will default to
	// P_EXTRA_SMALL
	//   "P_EXTRA_SMALL" - Extra small padding
	//   "P_SMALL" - Small padding
	//   "P_MEDIUM" - Medium padding
	//   "P_LARGE" - Large padding
	//   "P_EXTRA_LARGE" - Extra large padding
	Padding string `json:"padding,omitempty"`
	// PointerLocation: The pointer location for this widget (also sometimes called
	// a "tail")
	//
	// Possible values:
	//   "POINTER_LOCATION_UNSPECIFIED" - No visual pointer
	//   "PL_TOP" - Placed in the middle of the top of the widget
	//   "PL_RIGHT" - Placed in the middle of the right side of the widget
	//   "PL_BOTTOM" - Placed in the middle of the bottom of the widget
	//   "PL_LEFT" - Placed in the middle of the left side of the widget
	//   "PL_TOP_LEFT" - Placed on the left side of the top of the widget
	//   "PL_TOP_RIGHT" - Placed on the right side of the top of the widget
	//   "PL_RIGHT_TOP" - Placed on the top of the right side of the widget
	//   "PL_RIGHT_BOTTOM" - Placed on the bottom of the right side of the widget
	//   "PL_BOTTOM_RIGHT" - Placed on the right side of the bottom of the widget
	//   "PL_BOTTOM_LEFT" - Placed on the left side of the bottom of the widget
	//   "PL_LEFT_BOTTOM" - Placed on the bottom of the left side of the widget
	//   "PL_LEFT_TOP" - Placed on the top of the left side of the widget
	PointerLocation string `json:"pointerLocation,omitempty"`
	// TextColor: The text color as a hex string. "#RRGGBB" or "#RGB"
	TextColor string `json:"textColor,omitempty"`
	// VerticalAlignment: The vertical alignment of both the title and content
	//
	// Possible values:
	//   "VERTICAL_ALIGNMENT_UNSPECIFIED" - No vertical alignment specified, will
	// default to V_TOP
	//   "V_TOP" - Top-align
	//   "V_CENTER" - Center-align
	//   "V_BOTTOM" - Bottom-align
	VerticalAlignment string `json:"verticalAlignment,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BackgroundColor") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundColor") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TextStyle) MarshalJSON() ([]byte, error) {
	type NoMethod TextStyle
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Threshold: Defines a threshold for categorizing time series values.
type Threshold struct {
	// Color: The state color for this threshold. Color is not allowed in a
	// XyChart.
	//
	// Possible values:
	//   "COLOR_UNSPECIFIED" - Color is unspecified. Not allowed in well-formed
	// requests.
	//   "YELLOW" - Crossing the threshold is "concerning" behavior.
	//   "RED" - Crossing the threshold is "emergency" behavior.
	Color string `json:"color,omitempty"`
	// Direction: The direction for the current threshold. Direction is not allowed
	// in a XyChart.
	//
	// Possible values:
	//   "DIRECTION_UNSPECIFIED" - Not allowed in well-formed requests.
	//   "ABOVE" - The threshold will be considered crossed if the actual value is
	// above the threshold value.
	//   "BELOW" - The threshold will be considered crossed if the actual value is
	// below the threshold value.
	Direction string `json:"direction,omitempty"`
	// Label: A label for the threshold.
	Label string `json:"label,omitempty"`
	// TargetAxis: The target axis to use for plotting the threshold. Target axis
	// is not allowed in a Scorecard.
	//
	// Possible values:
	//   "TARGET_AXIS_UNSPECIFIED" - The target axis was not specified. Defaults to
	// Y1.
	//   "Y1" - The y_axis (the right axis of chart).
	//   "Y2" - The y2_axis (the left axis of chart).
	TargetAxis string `json:"targetAxis,omitempty"`
	// Value: The value of the threshold. The value should be defined in the native
	// scale of the metric.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Color") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Color") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Threshold) MarshalJSON() ([]byte, error) {
	type NoMethod Threshold
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Threshold) UnmarshalJSON(data []byte) error {
	type NoMethod Threshold
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// Tile: A single tile in the mosaic. The placement and size of the tile are
// configurable.
type Tile struct {
	// Height: The height of the tile, measured in grid blocks. Tiles must have a
	// minimum height of 1.
	Height int64 `json:"height,omitempty"`
	// Widget: The informational widget contained in the tile. For example an
	// XyChart.
	Widget *Widget `json:"widget,omitempty"`
	// Width: The width of the tile, measured in grid blocks. Tiles must have a
	// minimum width of 1.
	Width int64 `json:"width,omitempty"`
	// XPos: The zero-indexed position of the tile in grid blocks relative to the
	// left edge of the grid. Tiles must be contained within the specified number
	// of columns. x_pos cannot be negative.
	XPos int64 `json:"xPos,omitempty"`
	// YPos: The zero-indexed position of the tile in grid blocks relative to the
	// top edge of the grid. y_pos cannot be negative.
	YPos int64 `json:"yPos,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Height") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Height") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Tile) MarshalJSON() ([]byte, error) {
	type NoMethod Tile
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeSeriesFilter: A filter that defines a subset of time series data that is
// displayed in a widget. Time series data is fetched using the ListTimeSeries
// (https://cloud.google.com/monitoring/api/ref_v3/rest/v3/projects.timeSeries/list)
// method.
type TimeSeriesFilter struct {
	// Aggregation: By default, the raw time series data is returned. Use this
	// field to combine multiple time series for different views of the data.
	Aggregation *Aggregation `json:"aggregation,omitempty"`
	// Filter: Required. The monitoring filter
	// (https://cloud.google.com/monitoring/api/v3/filters) that identifies the
	// metric types, resources, and projects to query.
	Filter string `json:"filter,omitempty"`
	// PickTimeSeriesFilter: Ranking based time series filter.
	PickTimeSeriesFilter *PickTimeSeriesFilter `json:"pickTimeSeriesFilter,omitempty"`
	// SecondaryAggregation: Apply a second aggregation after aggregation is
	// applied.
	SecondaryAggregation *Aggregation `json:"secondaryAggregation,omitempty"`
	// StatisticalTimeSeriesFilter: Statistics based time series filter. Note: This
	// field is deprecated and completely ignored by the API.
	StatisticalTimeSeriesFilter *StatisticalTimeSeriesFilter `json:"statisticalTimeSeriesFilter,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aggregation") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aggregation") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeSeriesFilter) MarshalJSON() ([]byte, error) {
	type NoMethod TimeSeriesFilter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeSeriesFilterRatio: A pair of time series filters that define a ratio
// computation. The output time series is the pair-wise division of each
// aligned element from the numerator and denominator time series.
type TimeSeriesFilterRatio struct {
	// Denominator: The denominator of the ratio.
	Denominator *RatioPart `json:"denominator,omitempty"`
	// Numerator: The numerator of the ratio.
	Numerator *RatioPart `json:"numerator,omitempty"`
	// PickTimeSeriesFilter: Ranking based time series filter.
	PickTimeSeriesFilter *PickTimeSeriesFilter `json:"pickTimeSeriesFilter,omitempty"`
	// SecondaryAggregation: Apply a second aggregation after the ratio is
	// computed.
	SecondaryAggregation *Aggregation `json:"secondaryAggregation,omitempty"`
	// StatisticalTimeSeriesFilter: Statistics based time series filter. Note: This
	// field is deprecated and completely ignored by the API.
	StatisticalTimeSeriesFilter *StatisticalTimeSeriesFilter `json:"statisticalTimeSeriesFilter,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Denominator") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Denominator") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeSeriesFilterRatio) MarshalJSON() ([]byte, error) {
	type NoMethod TimeSeriesFilterRatio
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeSeriesQuery: TimeSeriesQuery collects the set of supported methods for
// querying time series data from the Stackdriver metrics API.
type TimeSeriesQuery struct {
	// OpsAnalyticsQuery: Preview: A query used to fetch a time series, category
	// series, or numeric series with SQL. This is a preview feature and may be
	// subject to change before final release.
	OpsAnalyticsQuery *OpsAnalyticsQuery `json:"opsAnalyticsQuery,omitempty"`
	// OutputFullDuration: Optional. If set, Cloud Monitoring will treat the full
	// query duration as the alignment period so that there will be only 1 output
	// value.*Note: This could override the configured alignment period except for
	// the cases where a series of data points are expected, like - XyChart -
	// Scorecard's spark chart
	OutputFullDuration bool `json:"outputFullDuration,omitempty"`
	// PrometheusQuery: A query used to fetch time series with PromQL.
	PrometheusQuery string `json:"prometheusQuery,omitempty"`
	// TimeSeriesFilter: Filter parameters to fetch time series.
	TimeSeriesFilter *TimeSeriesFilter `json:"timeSeriesFilter,omitempty"`
	// TimeSeriesFilterRatio: Parameters to fetch a ratio between two time series
	// filters.
	TimeSeriesFilterRatio *TimeSeriesFilterRatio `json:"timeSeriesFilterRatio,omitempty"`
	// TimeSeriesQueryLanguage: A query used to fetch time series with MQL.
	TimeSeriesQueryLanguage string `json:"timeSeriesQueryLanguage,omitempty"`
	// UnitOverride: The unit of data contained in fetched time series. If
	// non-empty, this unit will override any unit that accompanies fetched data.
	// The format is the same as the unit
	// (https://cloud.google.com/monitoring/api/ref_v3/rest/v3/projects.metricDescriptors)
	// field in MetricDescriptor.
	UnitOverride string `json:"unitOverride,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OpsAnalyticsQuery") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OpsAnalyticsQuery") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeSeriesQuery) MarshalJSON() ([]byte, error) {
	type NoMethod TimeSeriesQuery
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeSeriesTable: A table that displays time series data.
type TimeSeriesTable struct {
	// ColumnSettings: Optional. The list of the persistent column settings for the
	// table.
	ColumnSettings []*ColumnSettings `json:"columnSettings,omitempty"`
	// DataSets: Required. The data displayed in this table.
	DataSets []*TableDataSet `json:"dataSets,omitempty"`
	// MetricVisualization: Optional. Store rendering strategy
	//
	// Possible values:
	//   "METRIC_VISUALIZATION_UNSPECIFIED" - Unspecified state
	//   "NUMBER" - Default text rendering
	//   "BAR" - Horizontal bar rendering
	MetricVisualization string `json:"metricVisualization,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ColumnSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ColumnSettings") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeSeriesTable) MarshalJSON() ([]byte, error) {
	type NoMethod TimeSeriesTable
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Type: A protocol buffer message type.
type Type struct {
	// Edition: The source edition string, only valid when syntax is
	// SYNTAX_EDITIONS.
	Edition string `json:"edition,omitempty"`
	// Fields: The list of fields.
	Fields []*Field `json:"fields,omitempty"`
	// Name: The fully qualified message name.
	Name string `json:"name,omitempty"`
	// Oneofs: The list of types appearing in oneof definitions in this type.
	Oneofs []string `json:"oneofs,omitempty"`
	// Options: The protocol buffer options.
	Options []*Option `json:"options,omitempty"`
	// SourceContext: The source context.
	SourceContext *SourceContext `json:"sourceContext,omitempty"`
	// Syntax: The source syntax.
	//
	// Possible values:
	//   "SYNTAX_PROTO2" - Syntax proto2.
	//   "SYNTAX_PROTO3" - Syntax proto3.
	//   "SYNTAX_EDITIONS" - Syntax editions.
	Syntax string `json:"syntax,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Edition") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Edition") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Type) MarshalJSON() ([]byte, error) {
	type NoMethod Type
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Widget: Widget contains a single dashboard component and configuration of
// how to present the component in the dashboard.
type Widget struct {
	// AlertChart: A chart of alert policy data.
	AlertChart *AlertChart `json:"alertChart,omitempty"`
	// Blank: A blank space.
	Blank *Empty `json:"blank,omitempty"`
	// CollapsibleGroup: A widget that groups the other widgets. All widgets that
	// are within the area spanned by the grouping widget are considered member
	// widgets.
	CollapsibleGroup *CollapsibleGroup `json:"collapsibleGroup,omitempty"`
	// ErrorReportingPanel: A widget that displays a list of error groups.
	ErrorReportingPanel *ErrorReportingPanel `json:"errorReportingPanel,omitempty"`
	// Id: Optional. The widget id. Ids may be made up of alphanumerics, dashes and
	// underscores. Widget ids are optional.
	Id string `json:"id,omitempty"`
	// IncidentList: A widget that shows list of incidents.
	IncidentList *IncidentList `json:"incidentList,omitempty"`
	// LogsPanel: A widget that shows a stream of logs.
	LogsPanel *LogsPanel `json:"logsPanel,omitempty"`
	// PieChart: A widget that displays timeseries data as a pie chart.
	PieChart *PieChart `json:"pieChart,omitempty"`
	// Scorecard: A scorecard summarizing time series data.
	Scorecard *Scorecard `json:"scorecard,omitempty"`
	// SectionHeader: A widget that defines a section header for easier navigation
	// of the dashboard.
	SectionHeader *SectionHeader `json:"sectionHeader,omitempty"`
	// SingleViewGroup: A widget that groups the other widgets by using a dropdown
	// menu.
	SingleViewGroup *SingleViewGroup `json:"singleViewGroup,omitempty"`
	// Text: A raw string or markdown displaying textual content.
	Text *Text `json:"text,omitempty"`
	// TimeSeriesTable: A widget that displays time series data in a tabular
	// format.
	TimeSeriesTable *TimeSeriesTable `json:"timeSeriesTable,omitempty"`
	// Title: Optional. The title of the widget.
	Title string `json:"title,omitempty"`
	// XyChart: A chart of time series data.
	XyChart *XyChart `json:"xyChart,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AlertChart") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AlertChart") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Widget) MarshalJSON() ([]byte, error) {
	type NoMethod Widget
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// XyChart: A chart that displays data on a 2D (X and Y axes) plane.
type XyChart struct {
	// ChartOptions: Display options for the chart.
	ChartOptions *ChartOptions `json:"chartOptions,omitempty"`
	// DataSets: Required. The data displayed in this chart.
	DataSets []*DataSet `json:"dataSets,omitempty"`
	// Thresholds: Threshold lines drawn horizontally across the chart.
	Thresholds []*Threshold `json:"thresholds,omitempty"`
	// TimeshiftDuration: The duration used to display a comparison chart. A
	// comparison chart simultaneously shows values from two similar-length time
	// periods (e.g., week-over-week metrics). The duration must be positive, and
	// it can only be applied to charts with data sets of LINE plot type.
	TimeshiftDuration string `json:"timeshiftDuration,omitempty"`
	// XAxis: The properties applied to the x-axis.
	XAxis *Axis `json:"xAxis,omitempty"`
	// Y2Axis: The properties applied to the y2-axis.
	Y2Axis *Axis `json:"y2Axis,omitempty"`
	// YAxis: The properties applied to the y-axis.
	YAxis *Axis `json:"yAxis,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChartOptions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChartOptions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s XyChart) MarshalJSON() ([]byte, error) {
	type NoMethod XyChart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type LocationsGlobalMetricsScopesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns a specific Metrics Scope, including the list of projects
// monitored by the specified Metrics Scope.
//
//   - name: The resource name of the Metrics Scope. Example:
//     locations/global/metricsScopes/{SCOPING_PROJECT_ID_OR_NUMBER}.
func (r *LocationsGlobalMetricsScopesService) Get(name string) *LocationsGlobalMetricsScopesGetCall {
	c := &LocationsGlobalMetricsScopesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *LocationsGlobalMetricsScopesGetCall) Fields(s ...googleapi.Field) *LocationsGlobalMetricsScopesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *LocationsGlobalMetricsScopesGetCall) IfNoneMatch(entityTag string) *LocationsGlobalMetricsScopesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *LocationsGlobalMetricsScopesGetCall) Context(ctx context.Context) *LocationsGlobalMetricsScopesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *LocationsGlobalMetricsScopesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *LocationsGlobalMetricsScopesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.locations.global.metricsScopes.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *MetricsScope.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *LocationsGlobalMetricsScopesGetCall) Do(opts ...googleapi.CallOption) (*MetricsScope, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &MetricsScope{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListMetricsScopesByMonitoredProject: Returns a list of every Metrics Scope
// that a specific MonitoredProject has been added to. The metrics scope
// representing the specified monitored project will always be the first entry
// in the response.
func (r *LocationsGlobalMetricsScopesService) ListMetricsScopesByMonitoredProject() *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall {
	c := &LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// MonitoredResourceContainer sets the optional parameter
// "monitoredResourceContainer": Required. The resource name of the Monitored
// Project being requested. Example: projects/{MONITORED_PROJECT_ID_OR_NUMBER}
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) MonitoredResourceContainer(monitoredResourceContainer string) *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall {
	c.urlParams_.Set("monitoredResourceContainer", monitoredResourceContainer)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) Fields(s ...googleapi.Field) *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) IfNoneMatch(entityTag string) *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) Context(ctx context.Context) *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/locations/global/metricsScopes:listMetricsScopesByMonitoredProject")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.listMetricsScopesByMonitoredProject", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.locations.global.metricsScopes.listMetricsScopesByMonitoredProject" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListMetricsScopesByMonitoredProjectResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *LocationsGlobalMetricsScopesListMetricsScopesByMonitoredProjectCall) Do(opts ...googleapi.CallOption) (*ListMetricsScopesByMonitoredProjectResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListMetricsScopesByMonitoredProjectResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.listMetricsScopesByMonitoredProject", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type LocationsGlobalMetricsScopesProjectsCreateCall struct {
	s                *Service
	parent           string
	monitoredproject *MonitoredProject
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// Create: Adds a MonitoredProject with the given project ID to the specified
// Metrics Scope.
//
//   - parent: The resource name of the existing Metrics Scope that will monitor
//     this project. Example:
//     locations/global/metricsScopes/{SCOPING_PROJECT_ID_OR_NUMBER}.
func (r *LocationsGlobalMetricsScopesProjectsService) Create(parent string, monitoredproject *MonitoredProject) *LocationsGlobalMetricsScopesProjectsCreateCall {
	c := &LocationsGlobalMetricsScopesProjectsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.monitoredproject = monitoredproject
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *LocationsGlobalMetricsScopesProjectsCreateCall) Fields(s ...googleapi.Field) *LocationsGlobalMetricsScopesProjectsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *LocationsGlobalMetricsScopesProjectsCreateCall) Context(ctx context.Context) *LocationsGlobalMetricsScopesProjectsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *LocationsGlobalMetricsScopesProjectsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *LocationsGlobalMetricsScopesProjectsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.monitoredproject)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/projects")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.projects.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.locations.global.metricsScopes.projects.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *LocationsGlobalMetricsScopesProjectsCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.projects.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type LocationsGlobalMetricsScopesProjectsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a MonitoredProject from the specified Metrics Scope.
//
//   - name: The resource name of the MonitoredProject. Example:
//     locations/global/metricsScopes/{SCOPING_PROJECT_ID_OR_NUMBER}/projects/{MON
//     ITORED_PROJECT_ID_OR_NUMBER}Authorization requires the following Google
//     IAM (https://cloud.google.com/iam) permissions on both the Metrics Scope
//     and on the MonitoredProject: monitoring.metricsScopes.link.
func (r *LocationsGlobalMetricsScopesProjectsService) Delete(name string) *LocationsGlobalMetricsScopesProjectsDeleteCall {
	c := &LocationsGlobalMetricsScopesProjectsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *LocationsGlobalMetricsScopesProjectsDeleteCall) Fields(s ...googleapi.Field) *LocationsGlobalMetricsScopesProjectsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *LocationsGlobalMetricsScopesProjectsDeleteCall) Context(ctx context.Context) *LocationsGlobalMetricsScopesProjectsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *LocationsGlobalMetricsScopesProjectsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *LocationsGlobalMetricsScopesProjectsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.projects.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.locations.global.metricsScopes.projects.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *LocationsGlobalMetricsScopesProjectsDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.locations.global.metricsScopes.projects.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OperationsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest state of a long-running operation. Clients can use this
// method to poll the operation result at intervals as recommended by the API
// service.
//
// - name: The name of the operation resource.
func (r *OperationsService) Get(name string) *OperationsGetCall {
	c := &OperationsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsGetCall) Fields(s ...googleapi.Field) *OperationsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OperationsGetCall) IfNoneMatch(entityTag string) *OperationsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsGetCall) Context(ctx context.Context) *OperationsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.operations.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.operations.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsGetCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.operations.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsDashboardsCreateCall struct {
	s          *Service
	parent     string
	dashboard  *Dashboard
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a new custom dashboard. For examples on how you can use this
// API to create dashboards, see Managing dashboards by API
// (https://cloud.google.com/monitoring/dashboards/api-dashboard). This method
// requires the monitoring.dashboards.create permission on the specified
// project. For more information about permissions, see Cloud Identity and
// Access Management (https://cloud.google.com/iam).
//
//   - parent: The project on which to execute the request. The format is:
//     projects/[PROJECT_ID_OR_NUMBER] The [PROJECT_ID_OR_NUMBER] must match the
//     dashboard resource name.
func (r *ProjectsDashboardsService) Create(parent string, dashboard *Dashboard) *ProjectsDashboardsCreateCall {
	c := &ProjectsDashboardsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.dashboard = dashboard
	return c
}

// ValidateOnly sets the optional parameter "validateOnly": If set, validate
// the request and preview the review, but do not actually save it.
func (c *ProjectsDashboardsCreateCall) ValidateOnly(validateOnly bool) *ProjectsDashboardsCreateCall {
	c.urlParams_.Set("validateOnly", fmt.Sprint(validateOnly))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDashboardsCreateCall) Fields(s ...googleapi.Field) *ProjectsDashboardsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDashboardsCreateCall) Context(ctx context.Context) *ProjectsDashboardsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDashboardsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDashboardsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.dashboard)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/dashboards")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.dashboards.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Dashboard.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsDashboardsCreateCall) Do(opts ...googleapi.CallOption) (*Dashboard, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Dashboard{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsDashboardsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an existing custom dashboard.This method requires the
// monitoring.dashboards.delete permission on the specified dashboard. For more
// information, see Cloud Identity and Access Management
// (https://cloud.google.com/iam).
//
//   - name: The resource name of the Dashboard. The format is:
//     projects/[PROJECT_ID_OR_NUMBER]/dashboards/[DASHBOARD_ID].
func (r *ProjectsDashboardsService) Delete(name string) *ProjectsDashboardsDeleteCall {
	c := &ProjectsDashboardsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDashboardsDeleteCall) Fields(s ...googleapi.Field) *ProjectsDashboardsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDashboardsDeleteCall) Context(ctx context.Context) *ProjectsDashboardsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDashboardsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDashboardsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.dashboards.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsDashboardsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsDashboardsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Fetches a specific dashboard.This method requires the
// monitoring.dashboards.get permission on the specified dashboard. For more
// information, see Cloud Identity and Access Management
// (https://cloud.google.com/iam).
//
//   - name: The resource name of the Dashboard. The format is one of:
//     dashboards/[DASHBOARD_ID] (for system dashboards)
//     projects/[PROJECT_ID_OR_NUMBER]/dashboards/[DASHBOARD_ID] (for custom
//     dashboards).
func (r *ProjectsDashboardsService) Get(name string) *ProjectsDashboardsGetCall {
	c := &ProjectsDashboardsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDashboardsGetCall) Fields(s ...googleapi.Field) *ProjectsDashboardsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsDashboardsGetCall) IfNoneMatch(entityTag string) *ProjectsDashboardsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDashboardsGetCall) Context(ctx context.Context) *ProjectsDashboardsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDashboardsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDashboardsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.dashboards.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Dashboard.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsDashboardsGetCall) Do(opts ...googleapi.CallOption) (*Dashboard, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Dashboard{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsDashboardsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the existing dashboards.This method requires the
// monitoring.dashboards.list permission on the specified project. For more
// information, see Cloud Identity and Access Management
// (https://cloud.google.com/iam).
//
//   - parent: The scope of the dashboards to list. The format is:
//     projects/[PROJECT_ID_OR_NUMBER].
func (r *ProjectsDashboardsService) List(parent string) *ProjectsDashboardsListCall {
	c := &ProjectsDashboardsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": A positive number that is
// the maximum number of results to return. If unspecified, a default of 1000
// is used.
func (c *ProjectsDashboardsListCall) PageSize(pageSize int64) *ProjectsDashboardsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": If this field is not
// empty then it must contain the nextPageToken value returned by a previous
// call to this method. Using this field causes the method to return additional
// results from the previous method call.
func (c *ProjectsDashboardsListCall) PageToken(pageToken string) *ProjectsDashboardsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDashboardsListCall) Fields(s ...googleapi.Field) *ProjectsDashboardsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsDashboardsListCall) IfNoneMatch(entityTag string) *ProjectsDashboardsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDashboardsListCall) Context(ctx context.Context) *ProjectsDashboardsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDashboardsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDashboardsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/dashboards")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.dashboards.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListDashboardsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsDashboardsListCall) Do(opts ...googleapi.CallOption) (*ListDashboardsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListDashboardsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsDashboardsListCall) Pages(ctx context.Context, f func(*ListDashboardsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsDashboardsPatchCall struct {
	s          *Service
	name       string
	dashboard  *Dashboard
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Replaces an existing custom dashboard with a new definition.This
// method requires the monitoring.dashboards.update permission on the specified
// dashboard. For more information, see Cloud Identity and Access Management
// (https://cloud.google.com/iam).
//
// - name: Identifier. The resource name of the dashboard.
func (r *ProjectsDashboardsService) Patch(name string, dashboard *Dashboard) *ProjectsDashboardsPatchCall {
	c := &ProjectsDashboardsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.dashboard = dashboard
	return c
}

// ValidateOnly sets the optional parameter "validateOnly": If set, validate
// the request and preview the review, but do not actually save it.
func (c *ProjectsDashboardsPatchCall) ValidateOnly(validateOnly bool) *ProjectsDashboardsPatchCall {
	c.urlParams_.Set("validateOnly", fmt.Sprint(validateOnly))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDashboardsPatchCall) Fields(s ...googleapi.Field) *ProjectsDashboardsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDashboardsPatchCall) Context(ctx context.Context) *ProjectsDashboardsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDashboardsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDashboardsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.dashboard)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.dashboards.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Dashboard.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsDashboardsPatchCall) Do(opts ...googleapi.CallOption) (*Dashboard, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Dashboard{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.dashboards.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1LabelsCall struct {
	s                  *Service
	name               string
	location           string
	querylabelsrequest *QueryLabelsRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Labels: Lists labels for metrics.
//
//   - location: Location of the resource information. Has to be "global" now.
//   - name: The workspace on which to execute the request. It is not part of the
//     open source API but used as a request path prefix to distinguish different
//     virtual Prometheus instances of Google Prometheus Engine. The format is:
//     projects/PROJECT_ID_OR_NUMBER.
func (r *ProjectsLocationPrometheusApiV1Service) Labels(name string, location string, querylabelsrequest *QueryLabelsRequest) *ProjectsLocationPrometheusApiV1LabelsCall {
	c := &ProjectsLocationPrometheusApiV1LabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.querylabelsrequest = querylabelsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1LabelsCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1LabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1LabelsCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1LabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1LabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1LabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.querylabelsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/labels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.labels", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.labels" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1LabelsCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.labels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1QueryCall struct {
	s                   *Service
	name                string
	location            string
	queryinstantrequest *QueryInstantRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// Query: Evaluate a PromQL query at a single point in time.
//
//   - location: Location of the resource information. Has to be "global" now.
//   - name: The project on which to execute the request. Data associcated with
//     the project's workspace stored under the The format is:
//     projects/PROJECT_ID_OR_NUMBER. Open source API but used as a request path
//     prefix to distinguish different virtual Prometheus instances of Google
//     Prometheus Engine.
func (r *ProjectsLocationPrometheusApiV1Service) Query(name string, location string, queryinstantrequest *QueryInstantRequest) *ProjectsLocationPrometheusApiV1QueryCall {
	c := &ProjectsLocationPrometheusApiV1QueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.queryinstantrequest = queryinstantrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1QueryCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1QueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1QueryCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1QueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1QueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1QueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.queryinstantrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/query")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.query" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1QueryCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1QueryExemplarsCall struct {
	s                     *Service
	name                  string
	location              string
	queryexemplarsrequest *QueryExemplarsRequest
	urlParams_            gensupport.URLParams
	ctx_                  context.Context
	header_               http.Header
}

// QueryExemplars: Lists exemplars relevant to a given PromQL query,
//
//   - location: Location of the resource information. Has to be "global" now.
//   - name: The project on which to execute the request. Data associcated with
//     the project's workspace stored under the The format is:
//     projects/PROJECT_ID_OR_NUMBER. Open source API but used as a request path
//     prefix to distinguish different virtual Prometheus instances of Google
//     Prometheus Engine.
func (r *ProjectsLocationPrometheusApiV1Service) QueryExemplars(name string, location string, queryexemplarsrequest *QueryExemplarsRequest) *ProjectsLocationPrometheusApiV1QueryExemplarsCall {
	c := &ProjectsLocationPrometheusApiV1QueryExemplarsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.queryexemplarsrequest = queryexemplarsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1QueryExemplarsCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1QueryExemplarsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1QueryExemplarsCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1QueryExemplarsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1QueryExemplarsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1QueryExemplarsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.queryexemplarsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/query_exemplars")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query_exemplars", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.query_exemplars" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1QueryExemplarsCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query_exemplars", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1QueryRangeCall struct {
	s                 *Service
	name              string
	location          string
	queryrangerequest *QueryRangeRequest
	urlParams_        gensupport.URLParams
	ctx_              context.Context
	header_           http.Header
}

// QueryRange: Evaluate a PromQL query with start, end time range.
//
//   - location: Location of the resource information. Has to be "global" now.
//   - name: The project on which to execute the request. Data associcated with
//     the project's workspace stored under the The format is:
//     projects/PROJECT_ID_OR_NUMBER. Open source API but used as a request path
//     prefix to distinguish different virtual Prometheus instances of Google
//     Prometheus Engine.
func (r *ProjectsLocationPrometheusApiV1Service) QueryRange(name string, location string, queryrangerequest *QueryRangeRequest) *ProjectsLocationPrometheusApiV1QueryRangeCall {
	c := &ProjectsLocationPrometheusApiV1QueryRangeCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.queryrangerequest = queryrangerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1QueryRangeCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1QueryRangeCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1QueryRangeCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1QueryRangeCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1QueryRangeCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1QueryRangeCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.queryrangerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/query_range")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query_range", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.query_range" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1QueryRangeCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.query_range", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1SeriesCall struct {
	s                  *Service
	name               string
	location           string
	queryseriesrequest *QuerySeriesRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Series: Lists metadata for metrics.
//
//   - location: Location of the resource information. Has to be "global" for
//     now.
//   - name: The workspace on which to execute the request. It is not part of the
//     open source API but used as a request path prefix to distinguish different
//     virtual Prometheus instances of Google Prometheus Engine. The format is:
//     projects/PROJECT_ID_OR_NUMBER.
func (r *ProjectsLocationPrometheusApiV1Service) Series(name string, location string, queryseriesrequest *QuerySeriesRequest) *ProjectsLocationPrometheusApiV1SeriesCall {
	c := &ProjectsLocationPrometheusApiV1SeriesCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.queryseriesrequest = queryseriesrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1SeriesCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1SeriesCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1SeriesCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1SeriesCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1SeriesCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1SeriesCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.queryseriesrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/series")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.series", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.series" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1SeriesCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.series", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1LabelValuesCall struct {
	s            *Service
	name         string
	location     string
	label        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Values: Lists possible values for a given label name.
//
//   - label: The label name for which values are queried.
//   - location: Location of the resource information. Has to be "global" now.
//   - name: The workspace on which to execute the request. It is not part of the
//     open source API but used as a request path prefix to distinguish different
//     virtual Prometheus instances of Google Prometheus Engine. The format is:
//     projects/PROJECT_ID_OR_NUMBER.
func (r *ProjectsLocationPrometheusApiV1LabelService) Values(name string, location string, label string) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c := &ProjectsLocationPrometheusApiV1LabelValuesCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	c.label = label
	return c
}

// End sets the optional parameter "end": The end time to evaluate the query
// for. Either floating point UNIX seconds or RFC3339 formatted timestamp.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) End(end string) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.urlParams_.Set("end", end)
	return c
}

// Match sets the optional parameter "match": A list of matchers encoded in the
// Prometheus label matcher format to constrain the values to series that
// satisfy them.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Match(match string) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.urlParams_.Set("match", match)
	return c
}

// Start sets the optional parameter "start": The start time to evaluate the
// query for. Either floating point UNIX seconds or RFC3339 formatted
// timestamp.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Start(start string) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.urlParams_.Set("start", start)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) IfNoneMatch(entityTag string) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1LabelValuesCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/label/{label}/values")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
		"label":    c.label,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.label.values", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.label.values" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1LabelValuesCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.label.values", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationPrometheusApiV1MetadataListCall struct {
	s            *Service
	name         string
	location     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists metadata for metrics.
//
//   - location: Location of the resource information. Has to be "global" for
//     now.
//   - name: The workspace on which to execute the request. It is not part of the
//     open source API but used as a request path prefix to distinguish different
//     virtual Prometheus instances of Google Prometheus Engine. The format is:
//     projects/PROJECT_ID_OR_NUMBER.
func (r *ProjectsLocationPrometheusApiV1MetadataService) List(name string, location string) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c := &ProjectsLocationPrometheusApiV1MetadataListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.location = location
	return c
}

// Limit sets the optional parameter "limit": Maximum number of metrics to
// return.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Limit(limit int64) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c.urlParams_.Set("limit", fmt.Sprint(limit))
	return c
}

// Metric sets the optional parameter "metric": The metric name for which to
// query metadata. If unset, all metric metadata is returned.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Metric(metric string) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c.urlParams_.Set("metric", metric)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Fields(s ...googleapi.Field) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) IfNoneMatch(entityTag string) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Context(ctx context.Context) *ProjectsLocationPrometheusApiV1MetadataListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationPrometheusApiV1MetadataListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/location/{location}/prometheus/api/v1/metadata")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name":     c.name,
		"location": c.location,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.metadata.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "monitoring.projects.location.prometheus.api.v1.metadata.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *HttpBody.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationPrometheusApiV1MetadataListCall) Do(opts ...googleapi.CallOption) (*HttpBody, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &HttpBody{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "monitoring.projects.location.prometheus.api.v1.metadata.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
